package com.xebialabs.xlrelease.reports.excel

// scalastyle:off illegal.imports
import java.awt._
// scalastyle:on illegal.imports
import com.google.common.base.Strings
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.status.FlagStatus.{ATTENTION_NEEDED, AT_RISK, OK}
import com.xebialabs.xlrelease.planner.{Planner, PlannerReleaseItem, PlannerReleaseTree}
import com.xebialabs.xlrelease.reports.audit.CommonFormat.TaskAgent
import com.xebialabs.xlrelease.variable.VariableHelper.replaceAllWithInterpolation
import org.apache.poi.ss.usermodel.Workbook
import org.joda.time.DateTime.now

import java.util
import scala.jdk.CollectionConverters._

object ReleaseReport {
  private val SCHEDULED_START_DATE_HEADER = "Scheduled start date"
  private val DUE_DATE_HEADER = "Due date"
  private val START_DATE_HEADER = "Start date"
  private val END_DATE_HEADER = "End date"
  private val DURATION_HEADER = "Duration"
  private val ASSIGNEE_HEADER = "Assignee"
  private val TEAM_HEADER = "Team"
  private val COMMENTS_HEADER = "Task comments"
  private val FLAG_COMMENT_HEADER = "Status comment"
  private val DELAYED_HEADER = "Delayed"
  private val FAILURE_COUNT_HEADER = "Number of failure"
  private val MAXIMUM_CELL_LENGTH = 32767

  case class Data(release: Release, taskAgentsData: Map[String, TaskAgent])

  def javaData(release: Release, tasksAgentsData: java.util.Map[String, TaskAgent]): Data =
    Data(release, tasksAgentsData.asScala.toMap)
}

class ReleaseReport(data: ReleaseReport.Data) {
  private var tree: PlannerReleaseTree = _
  private var sheetWriter: ExcelSheetWriter = _

  def getWorkBook(): Workbook = {
    val report = new ReportWorkbook
    sheetWriter = report.createReportSection("overview")
    sheetWriter.newRow
    addHeaders()
    addContent()
    report.getWorkbookWithoutErrorSheet
  }

  private def addContent(): Unit = {
    tree = PlannerReleaseTree(PlannerReleaseItem.transform(data.release))
    Planner.makePlan(tree, now) // it updates our tree as side-effect.

    addReleaseRow()
    for (phase <- tree.root.getPhases().asScala) {
      addPhaseRow(phase)
      for (task <- phase.getAllTasks().asScala) {
        addTaskRow(task)
      }
    }
  }

  private def addHeaders(): Unit = {
    sheetWriter.addHeaderCell("", 30)
      .addHeaderCell("", 15)
      .addHeaderCell("", 15)
      .addHeaderCell(ReleaseReport.SCHEDULED_START_DATE_HEADER, 16)
      .addHeaderCell(ReleaseReport.DUE_DATE_HEADER, 15)
      .addHeaderCell(ReleaseReport.START_DATE_HEADER, 15)
      .addHeaderCell(ReleaseReport.END_DATE_HEADER, 15)
      .addHeaderCell(ReleaseReport.DURATION_HEADER, 15)
      .addHeaderCell(ReleaseReport.ASSIGNEE_HEADER, 15)
      .addHeaderCell(ReleaseReport.TEAM_HEADER, 15)
      .addHeaderCell(ReleaseReport.COMMENTS_HEADER, 25)
      .addHeaderCell(ReleaseReport.FLAG_COMMENT_HEADER, 20)
      .addHeaderCell(ReleaseReport.DELAYED_HEADER, 5)
      .addHeaderCell(ReleaseReport.FAILURE_COUNT_HEADER, 5)
      .newRow.newRow
  }

  private def addReleaseRow(): Unit = {
    sheetWriter.isBoldRow = true
    val plannedRelease = tree.root
    sheetWriter.addCell(data.release.getTitle)
    sheetWriter.indent
    sheetWriter.addCell(data.release.getStatus.toString)
    addScheduledStartDateCell(tree.root)
    addDueDateCell(plannedRelease)
    addStartDateCell(plannedRelease)
    addEndDateCell(plannedRelease)
    addDurationCell(plannedRelease)
    addFlagCommentCell(data.release)
    sheetWriter.isBoldRow = false
    sheetWriter.newRow
    sheetWriter.newRow
  }

  private def addPhaseRow(plannedPhase: PlannerReleaseItem): Unit = {
    val phase = plannedPhase.getItem().asInstanceOf[Phase]
    sheetWriter.isPhaseRow = true
    sheetWriter.applyPhaseColor(phase).addCell(getTitle(phase)).newColumn.addCell(phase.getStatus.toString, sheetWriter.getDateStyle)
    addScheduledStartDateCell(plannedPhase)
    addDueDateCell(plannedPhase)
    addStartDateCell(plannedPhase)
    addEndDateCell(plannedPhase)
    addDurationCell(plannedPhase)
    sheetWriter.newColumn.newColumn.newColumn.newColumn.newColumn.newColumn.newRow
    sheetWriter.isPhaseRow = false
  }

  private def addTaskRow(plannedTask: PlannerReleaseItem): Unit = {
    val task = plannedTask.getItem().asInstanceOf[Task]
    sheetWriter.addCell(getTitle(task))
    sheetWriter.addCell(getTaskType(task), sheetWriter.getDateStyle)
    sheetWriter.addCell(task.getStatus.toString, sheetWriter.getDateStyle)
    addScheduledStartDateCell(plannedTask)
    addDueDateCell(plannedTask)
    addStartDateCell(plannedTask)
    addEndDateCell(plannedTask)
    addDurationCell(plannedTask)
    addAssigneeCell(task)
    addTeamCell(task)
    addTaskComments(task)
    addFlagCommentCell(task)
    addDelayedCell(task)
    addFailureCountCell(task)
    sheetWriter.newRow
  }

  private def getTitle(task: Task) = {
    var title = task.getTitle
    var container = task.getContainer
    while ( {
      container.isInstanceOf[TaskGroup]
    }) {
      title = container.asInstanceOf[TaskGroup].getTitle + " / " + title
      container = container.asInstanceOf[TaskGroup].getContainer
    }
    title
  }

  private def getTitle(item: PlanItem): String = getTitle(item.getTitle)

  private def getTitle(title: String): String = replaceAllWithInterpolation(title, data.release.getAllVariableValuesAsStringsWithInterpolationInfo, new util.HashSet[String], freezeEvenIfUnresolved = false)

  private def addScheduledStartDateCell(item: PlannerReleaseItem): Unit = {
    val date = item.getScheduledStartDate
    sheetWriter.addCell(date)
  }

  private def addDueDateCell(item: PlannerReleaseItem): Unit = {
    val date = item.getDueDate
    sheetWriter.addCell(date)
  }

  private def addStartDateCell(item: PlannerReleaseItem): Unit = {
    val date = item.getStartDate
    sheetWriter.addCell(date)
  }

  private def addEndDateCell(item: PlannerReleaseItem): Unit = {
    val date = item.getEndDate
    sheetWriter.addCell(date)
  }

  private def addDurationCell(item: PlannerReleaseItem): Unit = {
    val duration = item.getDuration
    sheetWriter.addCell(duration)
  }

  private def addAssigneeCell(task: Task): Unit = {
    sheetWriter.addCell(data.taskAgentsData.get(task.getId).flatMap(_.assignedUserAndTeam._1).orNull)
  }

  private def addTeamCell(task: Task): Unit = {
    sheetWriter.addCell(task.getTeam)
  }

  private def addTaskComments(task: Task): Unit = {
    val comments = task.getComments.asScala.map((entity: Comment) => entity.getAuthor + ": " + entity.getText).mkString("\n")
    sheetWriter.addFullText(comments.substring(0, Math.min(comments.length, ReleaseReport.MAXIMUM_CELL_LENGTH)))
  }

  private def addFlagCommentCell(item: PlanItem): Unit = {
    if (item.getFlagStatus eq OK) {
      sheetWriter.indent
    } else {
      val backgroundColor: Color = item.getFlagStatus match {
        case ATTENTION_NEEDED => Color.ORANGE
        case AT_RISK => Color.RED
        case _ => null
      }
      sheetWriter.addCell(item.getFlagComment, backgroundColor)
    }
  }

  private def addDelayedCell(task: Task): Unit = {
    sheetWriter.addCell(
      if (task.getDelayedCount > 0) "Yes" else "No"
    )
  }

  private def addFailureCountCell(task: Task): Unit = {
    sheetWriter.addCell(String.valueOf(task.getFailuresCount))
  }

  private def getTaskType(task: Task) = {
    val typeLabel = task.getType.getDescriptor.getLabel
    if (Strings.isNullOrEmpty(typeLabel)) {
      task.getType.getName
    } else {
      typeLabel
    }
  }
}
