package com.xebialabs.xlrelease.reports.repository.itsm

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters
import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.udm.reporting.ItsmRecord
import com.xebialabs.xlrelease.domain.utils.FullReleaseId
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.reports.repository.CommonRowAndParamsMapper._
import com.xebialabs.xlrelease.reports.repository.{ArchivedTasksRepository, RepositoryExceptionUtils, SearchRecordsSupport}
import com.xebialabs.xlrelease.repository.FacetRepository
import com.xebialabs.xlrelease.repository.FacetRepository.SpecializedFacetRepository
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.collection.mutable
import scala.util.Failure

object ItsmRecordArchiveRepository extends Logging {

  import ITSM_TASK_REPORTING_RECORD_ARCHIVE._

  private lazy val ALL_FIELDS =
    s"""| ${commonFieldNames(ITSM_TASK_REPORTING_RECORD_ARCHIVE, "i")},
        | t.$REPORT_TASKS_ID_COLUMN,
        | i.$RECORD,
        | i.$RECORD_URL,
        | i.$TITLE,
        | i.$STATUS,
        | i.$PRIORITY,
        | i.$CREATED_BY""".stripMargin

  private lazy val SELECT_ALL_ITSM_RECORDS: String =
    s"""|SELECT
        | $ALL_FIELDS
        |FROM ${TABLE} i
        |JOIN $REPORT_TASKS_TABLE_NAME t ON t.$REPORT_TASKS_UID_COLUMN = i.$TASK_UID""".stripMargin

  private lazy val SELECT_ITSM_RECORD_BY_ID: String =
    s"""|$SELECT_ALL_ITSM_RECORDS
        |WHERE i.$RECORD_ID = :recordId""".stripMargin

  private lazy val SELECT_ITSM_RECORDS_BY_TASK_ID: String =
    s"""|$SELECT_ALL_ITSM_RECORDS
        |WHERE t.$REPORT_TASKS_ID_COLUMN = :taskId""".stripMargin

  private lazy val SELECT_ITSM_RECORDS_BY_RELEASE_ID: String =
    s"""|$SELECT_ALL_ITSM_RECORDS
        |JOIN $REPORT_RELEASES_TABLE_NAME r ON r.$REPORT_RELEASES_ID_COLUMN = t.$REPORT_TASKS_RELEASEID_COLUMN
        |WHERE r.$REPORT_RELEASES_ID_COLUMN = :releaseId AND r.$REPORT_RELEASES_PRE_ARCHIVED <> 1""".stripMargin

  private lazy val EXISTS_ITSM_RECORD_BY_ID: String =
    s"""SELECT COUNT($RECORD_ID) FROM ${TABLE} WHERE $RECORD_ID = :recordId""".stripMargin

  private lazy val INSERT_ITSM_RECORD: String =
    s"""|INSERT INTO ${TABLE} (
        | ${commonFieldNames(ITSM_TASK_REPORTING_RECORD_ARCHIVE)},
        | $TASK_UID,
        | $RECORD,
        | $RECORD_URL,
        | $TITLE,
        | $STATUS,
        | $PRIORITY,
        | $CREATED_BY
        |) VALUES (
        | ${commonParameterNames()},
        | :taskUid,
        | :record,
        | :recordUrl,
        | :title,
        | :status,
        | :priority,
        | :createdBy
        |)""".stripMargin

  private lazy val DELETE_ITSM_RECORD: String =
    s"""|DELETE FROM ${TABLE}
        |WHERE $RECORD_ID = :recordId""".stripMargin

}

class ItsmRecordArchiveRepository(archivedTasksRepository: ArchivedTasksRepository,
                                  val ciIdService: CiIdService,
                                  val jdbcTemplate: JdbcTemplate,
                                  val dialect: Dialect)
  extends SpecializedFacetRepository(Type.valueOf(classOf[ItsmRecord]))
    with FacetRepository.ForArchive
    with PersistenceSupport
    with SearchRecordsSupport[ItsmRecord]
    with RepositoryExceptionUtils
    with Logging {

  import ItsmRecordArchiveRepository._

  def get(recordId: String, recordType: Option[Type]): Facet = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(SELECT_ITSM_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), itsmRowMapper)
      .headOption.getOrElse(throw new NotFoundException(s"Record '$recordId' not found."))
  }

  override protected def doCreate(facet: Facet): Facet = {
    logger.debug("Creating ITSM record in archive")
    requiresInstanceOfSupportedType(facet.getType)
    archivedTasksRepository.taskUidById(shortenId(facet.getTargetId)).map { taskUid =>
      val itsmFacet = facet.asInstanceOf[ItsmRecord]
      sqlInsert(INSERT_ITSM_RECORD, asQueryParameters(itsmFacet, taskUid))
      facet
    }.recoverWith {
      case _: NotFoundException =>
        Failure(targetTaskNotFoundException(facet.getId, facet.getTargetId))
    }.get
  }

  def delete(recordId: String, recordType: Option[Type]): Unit =
    sqlUpdate(DELETE_ITSM_RECORD, params("recordId" -> getName(recordId.normalized)), _ => ())

  def update(record: Facet): Facet = {
    throw new UnsupportedOperationException("You cannot update records!")
  }

  def exists(recordId: String, recordType: Option[Type]): Boolean = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(EXISTS_ITSM_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), _.getInt(1) > 0).headOption.exists(identity)
  }

  def findAllFacetsByRelease(release: Release): Seq[Facet] = {
    findAllFacetsByReleaseId(release.getId).toSeq
  }

  def findAllFacetsByTask(task: Task): Seq[Facet] = {
    findAllFacetsByTargetId(shortenId(task.getId)).toSeq
  }

  def search(facetsFilters: FacetFilters): Seq[Facet] = searchRecords(facetsFilters, SELECT_ALL_ITSM_RECORDS, itsmRowMapper)

  override protected def findAllFacetsByTargetId(targetId: CiId): mutable.Seq[ItsmRecord] =
    sqlQuery(SELECT_ITSM_RECORDS_BY_TASK_ID, params("taskId" -> targetId), itsmRowMapper)

  override protected def findAllFacetsByReleaseId(releaseId: CiId): mutable.Seq[ItsmRecord] = {
    sqlQuery(SELECT_ITSM_RECORDS_BY_RELEASE_ID, params(
      "releaseId" -> FullReleaseId(releaseId).withOnlyOneParentOrApplicationsForArchiveDb()
    ), itsmRowMapper)
  }

  private val itsmRowMapper: RowMapper[ItsmRecord] = (rs: ResultSet, _: Int) => {
    val itsmRecord: ItsmRecord = Type.valueOf(rs.getString(ITSM_TASK_REPORTING_RECORD_ARCHIVE.RECORD_TYPE))
      .getDescriptor.newInstance[ItsmRecord](rs.getString(ITSM_TASK_REPORTING_RECORD_ARCHIVE.RECORD_ID))
    itsmEntryRowMapper(rs, itsmRecord, rs.getString(REPORT_TASKS_ID_COLUMN), ITSM_TASK_REPORTING_RECORD_ARCHIVE)
  }

}
