package com.xebialabs.xlrelease.reports.service


import com.github.benmanes.caffeine.cache.{Cache, LoadingCache}
import com.xebialabs.xlrelease.config.CacheManagementConstants.REPORTS_CACHE_MANAGER
import com.xebialabs.xlrelease.reports.dto._
import com.xebialabs.xlrelease.reports.service.ReportServiceCache.GetDataWithRefresh
import com.xebialabs.xlrelease.support.cache.caffeine.spring.XlrCaffeineCacheManager
import com.xebialabs.xlrelease.views.Point
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Service

import java.util.{List => JList}

object ReportServiceCache {

  implicit class GetDataWithRefresh[A](val cache: LoadingCache[ReportParams, A]) {
    def getData(params: ReportParams): A = {
      if (params.refresh) {
        cache.invalidate(params)
      }

      cache.get(params)
    }
  }
}

@Service
class ReportServiceCache @Autowired()(reportService: ReportService, @Qualifier(REPORTS_CACHE_MANAGER) cacheManager: XlrCaffeineCacheManager) {

  def builder[A <: Object](name: String, f: ReportParams => A): LoadingCache[ReportParams, A] = {
    cacheManager.createNativeCacheAndRegister(name, (key: ReportParams) => f(key))
  }

  import com.xebialabs.xlrelease.reports.config.ReportServiceCacheConfig._

  private val releaseDurationCache =
    builder(RELEASE_DURATION_CACHE, p => reportService.getReleaseDuration(p))

  private val completedReleasesCache =
    builder(COMPLETED_RELEASES_CACHE, p => reportService.getCompletedReleases(p))

  private val averageAndLongestReleaseDurationCache =
    builder(AVERAGE_AND_LONGEST_RELEASE_DURATION_CACHE, p => reportService.getAverageAndLongestReleaseDuration(p))

  private val peopleMostInvolvedCache =
    builder(PEOPLE_MOST_INVOLVED_CACHE, p => reportService.getTopPeopleMostInvolved(p))

  private val longestTasksCache =
    builder(LONGEST_TASKS_CACHE, p => reportService.getTopLongestTasks(p))

  private val longestTaskTypesCache =
    builder(LONGEST_TASK_TYPES_CACHE, p => reportService.getTopLongestTaskTypes(p))

  private val averageAndLongestTaskDurationCache =
    builder(AVERAGE_AND_LONGEST_TASK_DURATION_CACHE, p => reportService.getAverageAndLongestTaskDuration(p))

  private val releaseAutomationCache =
    builder(RELEASE_AUTOMATION_CACHE, p => reportService.getReleaseAutomationSeries(p))

  private val mostRecentReleasesCache =
    builder(MOST_RECENT_RELEASES_CACHE, p => reportService.getReleaseAutomation(p))

  private val longestPhasesCache =
    builder(LONGEST_PHASES_CACHE, p => reportService.getTopLongestPhases(p))

  private val numberOfReleaseCache =
    builder(NUMBER_OF_RELEASES_CACHE, p => reportService.getNumberOfReleaseByMonth(p))

  def getReleaseDuration(reportParams: ReportParams): ReleasesDuration = releaseDurationCache.getData(reportParams)

  def getCompletedReleases(reportParams: ReportParams): CompletedReleases = completedReleasesCache.getData(reportParams)

  def getAverageAndLongestReleaseDuration(reportParams: ReportParams): AverageAndLongestReleaseDuration =
    averageAndLongestReleaseDurationCache.getData(reportParams)

  def getTopPeopleMostInvolved(reportParams: ReportParams): JList[UserParticipation] = peopleMostInvolvedCache.getData(reportParams)

  def getTopLongestTasks(reportParams: ReportParams): JList[TaskDuration] = longestTasksCache.getData(reportParams)

  def getTopLongestTaskTypes(reportParams: ReportParams): JList[LongestTaskType] = longestTaskTypesCache.getData(reportParams)

  def getAverageAndLongestTaskDuration(reportParams: ReportParams): AverageAndLongestTaskDuration =
    averageAndLongestTaskDurationCache.getData(reportParams)

  def getReleaseAutomationSeries(reportParams: ReportParams): ReleasesAutomation = releaseAutomationCache.getData(reportParams)

  def getTopLongestPhases(reportParams: ReportParams): JList[PhaseDuration] = longestPhasesCache.getData(reportParams)

  def getNumberOfReleaseByMonth(reportParams: ReportParams): JList[Point] = numberOfReleaseCache.getData(reportParams)

  def getMostRecentReleasesData(reportParams: ReportParams): JList[ReleaseAutomationData] = mostRecentReleasesCache.getData(reportParams)

  private val cache = List[Cache[_, _]](
    releaseDurationCache, peopleMostInvolvedCache, longestTasksCache,
    releaseAutomationCache, longestPhasesCache, numberOfReleaseCache,
    mostRecentReleasesCache
  )

  def clearCache(): Unit = {
    cache.foreach(_.invalidateAll())
  }
}
