package com.xebialabs.xlrelease.reports.timeline

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.security.Permissions
import com.xebialabs.xlrelease.api.v1.forms.ReleasesFilters
import com.xebialabs.xlrelease.domain.calendar.{Blackout, SpecialDay}
import com.xebialabs.xlrelease.reports.audit.InstanceData
import com.xebialabs.xlrelease.reports.service.ReportsService
import com.xebialabs.xlrelease.reports.timeline.TimelineExport.ReleaseData
import com.xebialabs.xlrelease.reports.timeline.TimelineExportComponent.RELEASE_BUFFER_SIZE
import com.xebialabs.xlrelease.repository.CalendarEntryRepository
import com.xebialabs.xlrelease.search.ReleaseSearchResult
import com.xebialabs.xlrelease.service.{FolderService, SqlReleaseSearchService}
import grizzled.slf4j.Logging
import org.apache.poi.ss.usermodel.Workbook
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util.Date
import scala.collection.mutable
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._
import scala.util.Try

object TimelineExportComponent {
  val RELEASE_BUFFER_SIZE = 50
}

@Component
class TimelineExportComponent @Autowired()(reportsService: ReportsService,
                                           sqlReleaseSearchService: SqlReleaseSearchService,
                                           calendarEntryRepository: CalendarEntryRepository,
                                           folderService: FolderService) extends Logging {

  def exportTimeline(releasesFilters: ReleasesFilters): Workbook = {
    val instanceData = InstanceData(reportsService.getInstanceName(), reportsService.getServerUrl(), reportsService.getInstanceVersion())
    val generatedBy = Permissions.getAuthenticatedUserName
    val releasesData = getReleasesData(releasesFilters)
    val blackouts = getBlackoutData(releasesFilters.getFrom, releasesFilters.getTo)
    val labels = getSpecialDates(releasesFilters.getFrom, releasesFilters.getTo)
    val data = TimelineExport.Data(generatedBy, new Date(), instanceData, releasesData.toSeq, blackouts, labels)
    TimelineExport.getWorkBook(data, releasesFilters)
  }


  private def getReleasesData(releasesFilters: ReleasesFilters): mutable.Seq[ReleaseData] = {
    var releaseData = ListBuffer.empty[ReleaseData]
    var page = 0
    var releases: ReleaseSearchResult = sqlReleaseSearchService.search(releasesFilters, page, RELEASE_BUFFER_SIZE)
    while (releases.getSize > 0) {
      releases.getReleases.asScala.foreach { release =>
        val folderTitle = Try(folderService.findById(release.findFolderId())).toOption.map(_.getTitle).getOrElse("")
        releaseData += TimelineExport.ReleaseData(release.getTitle, release.getStatus, release.getScheduledStartDate,
          release.getDueDate, release.getStartDate, release.getEndDate, folderTitle, release.getPhases.asScala.toList)
      }
      page += 1
      releases = sqlReleaseSearchService.search(releasesFilters, page, RELEASE_BUFFER_SIZE)
    }
    releaseData
  }

  private def getBlackoutData(from: Date, to: Date): Seq[Blackout] =
    calendarEntryRepository.findAllByTypeInRange[Blackout](Type.valueOf(classOf[Blackout]), from, to).asScala.toSeq

  private def getSpecialDates(from: Date, to: Date): Seq[SpecialDay] =
    calendarEntryRepository.findAllByTypeInRange[SpecialDay](Type.valueOf(classOf[SpecialDay]), from, to).asScala.toSeq
}
