package com.xebialabs.xlrelease.udm.reporting.events

import com.xebialabs.xlrelease.api.internal.{DecoratorCache, ReleaseServerUrlDecorator}
import com.xebialabs.xlrelease.domain.Task
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.udm.reporting.{DeploymentRecord, DeploymentStatus}
import com.xebialabs.xlrelease.environments.repository.{ApplicationRepository, EnvironmentRepository}
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.reports.repository.deployment.DeploymentRecordRepository
import com.xebialabs.xlrelease.repository.{FacetRepositoryDispatcher, TaskRepository}
import com.xebialabs.xlrelease.service.{FacetService, FolderVariableService, VariableService}
import com.xebialabs.xlrelease.udm.reporting.DeploymentTaskFacet
import com.xebialabs.xlrelease.udm.reporting.repository.DeploymentRepository
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._
import scala.util.Try

@Component
@EventListener
class FacetDeploymentListener @Autowired()(val deploymentRepository: DeploymentRepository,
                                           val applicationRepository: ApplicationRepository,
                                           val environmentRepository: EnvironmentRepository,
                                           val taskRepository: TaskRepository,
                                           val variableService: VariableService,
                                           val folderVariableService: FolderVariableService,
                                           val facetRepositoryDispatcher: FacetRepositoryDispatcher,
                                           val facetService: FacetService,
                                           val releaseServerUrlDecorator: ReleaseServerUrlDecorator,
                                           val deploymentRecordRepository: DeploymentRecordRepository) extends DeploymentListener {
  @Subscribe
  def onFacetEvent(event: FacetEvent): Unit = event match {
    case FacetDeletedEvent(facet) if isDeploymentFacet(facet) =>
      deleteDeployment(facet)
    case _ if isDeploymentFacet(event.facet) =>
      Try(taskRepository.findById[Task](event.facet.getTargetId)).foreach { task =>
        val status = calculateDeploymentStatus(task)
        if (status.isDefined) {
          val unresolvedVariables = resolveVariables(event.facet, task.getRelease)
          if (unresolvedVariables.isEmpty) {
            saveDeployment(task, event.facet.asInstanceOf[DeploymentTaskFacet], status.get, autoGenerated = false)
          } else {
            logger.debug(s"Deployment event ${event.facet.getId} ignored due to unresolved variables $unresolvedVariables")
          }
        }
      }
    case _ => // nothing
  }

  @Subscribe
  def onTaskEvent(event: TaskExecutionEvent): Unit = {
    try {
      val status = calculateDeploymentStatus(event.task)
      val hasDeploymentFacets = event.task.getFacets.asScala.exists(isDeploymentFacet)
      if (status.isDefined && hasDeploymentFacets) {
        saveDeploymentsAndDeploymentRecords(event, status.get)
      }
    } catch {
      case any: Throwable =>
        logger.error(any.getMessage, any)
    }
  }

  @Subscribe
  def onFacetConfiguredFacetsCreatedEvent(event: FacetConfiguredFacetsCreatedEvent): Unit = {
    event.originalFacet match {
      case deploymentFacet: DeploymentRecord if
        // we only create IN_PROGRESS events in advance
        deploymentFacet.getStatus == DeploymentStatus.IN_PROGRESS &&
          // we only replace values that we created earlier if plugin or script itself create
          // proper facets via facetApi now
          deploymentFacet.isCreatedViaApi &&
          // we only replace values if task has configuration attributes
          event.hasConfiguration =>
        // we should remove all previous IN_PROGRESS facets that we created automatically
        // from attributes ourselves
        deploymentRecordRepository.liveRepository.deleteAutogeneratedInProgressFacets(
          deploymentFacet.getTargetId,
          deploymentFacet.getRetryAttemptNumber
        )
      case _ => // no action
    }
  }

  def saveDeploymentsAndDeploymentRecords(event: TaskExecutionEvent, status: DeploymentStatus): Unit = {
    val task = event.task
    val deploymentFacets = task.getFacets.asScala.filter(isDeploymentFacet)
    deploymentFacets.foreach(facet => saveDeployment(task, facet.asInstanceOf[DeploymentTaskFacet], status, autoGenerated = false))
    val failuresCount = calculateFailuresCount(event)

    status match {
      case DeploymentStatus.IN_PROGRESS =>
        val record = new DeploymentRecord()
        record.setRetryAttemptNumber(failuresCount)
        record.setStatus(DeploymentStatus.IN_PROGRESS)
        record.setTargetId(task.getId)
        record.setDeploymentTask(task.getId)
        record.setDeploymentTask_url(task.getUrl)
        facetService.createApplyingConfiguration(record, createUnconfigured = false)
      case deploymentStatus =>
        val hasExistingRecords = deploymentRecordRepository.liveRepository.facetsCreatedViaApiExist(task, failuresCount)
        if (!hasExistingRecords) {
          // ok, we have some configuration facets (aka Attributes) for this task, and
          // we can see, that task did not call facetApi to create immutable facets, so we should
          // do it
          releaseServerUrlDecorator.decorate(task.getRelease, new DecoratorCache[Unit])
          val record = new DeploymentRecord()
          record.setRetryAttemptNumber(failuresCount)
          record.setStatus(deploymentStatus)
          record.setTargetId(task.getId)
          record.setDeploymentTask(task.getId)
          record.setDeploymentTask_url(task.getUrl)
          facetService.createApplyingConfiguration(record, createUnconfigured = false)
        }
    }
  }

  def calculateFailuresCount(event: TaskExecutionEvent): Int = {
    event match {
      case taskFailedEvent: TaskFailedEvent => taskFailedEvent.task.getFailuresCount - 1
      case _ => event.task.getFailuresCount
    }
  }

  def deleteDeployment(facet: Facet): Unit = {
    logger.debug(s"Removing deployment events for facet $facet")
    deploymentRepository.delete(getDeploymentId(facet))
  }
}
