package com.xebialabs.xlrelease.reports.job.repository.sql

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.reports.job.api.ReportDefinition
import com.xebialabs.xlrelease.reports.job.api.ReportingEngineService.ReportJobId
import com.xebialabs.xlrelease.reports.job.domain.ReportJob
import com.xebialabs.xlrelease.reports.job.domain.ReportJobStatus.{STARTED, SUBMITTED}
import com.xebialabs.xlrelease.reports.job.repository.sql.ReportJobSchema.{REPORT_JOBS => RJ}
import com.xebialabs.xlrelease.reports.job.repository.sql.SqlReportJobRepository.ReportDefinitionExtensions
import com.xebialabs.xlrelease.reports.job.repository.{QueryBuilder, ReportJobFilters, ReportJobRepository}
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceConstants.BLOB_TYPE
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import grizzled.slf4j.Logging
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource
import org.springframework.jdbc.core.support.SqlBinaryValue

import java.nio.charset.StandardCharsets
import java.util.Date
import scala.jdk.CollectionConverters._

@IsTransactional
class SqlReportJobRepository(val jdbcTemplate: JdbcTemplate,
                             val dialect: Dialect,
                             implicit val sqlRepositoryAdapter: SqlRepositoryAdapter)
  extends ReportJobRepository with PersistenceSupport with Logging with ReportJobMapper {

  private val STMT_CREATE_REPORT_JOB =
    s"""INSERT INTO ${RJ.TABLE} (${RJ.REPORT_TYPE},
       | ${RJ.REPORT_NAME}, ${RJ.STATUS},
       | ${RJ.NODE},
       | ${RJ.TOTAL_WORK_ITEMS}, ${RJ.COMPLETED_WORK_ITEMS},
       | ${RJ.SUBMIT_TIME}, ${RJ.START_TIME}, ${RJ.END_TIME},
       | ${RJ.USERNAME}, ${RJ.RESULT_URI}, ${RJ.REPORT_DEFINITION})
       | VALUES (:${RJ.REPORT_TYPE},
       | :${RJ.REPORT_NAME}, :${RJ.STATUS},
       | :${RJ.NODE},
       | :${RJ.TOTAL_WORK_ITEMS}, :${RJ.COMPLETED_WORK_ITEMS},
       | :${RJ.SUBMIT_TIME}, :${RJ.START_TIME}, :${RJ.END_TIME},
       | :${RJ.USERNAME}, :${RJ.RESULT_URI}, :${RJ.REPORT_DEFINITION})
         """.stripMargin

  override def create(reportJob: ReportJob): ReportJob = {
    val params = new MapSqlParameterSource()
    params.addValue(RJ.REPORT_TYPE, reportJob.getReportDefinition.getType.toString)
    params.addValue(RJ.REPORT_NAME, reportJob.reportName.truncateBytes(755))
    params.addValue(RJ.STATUS, reportJob.status.name())
    params.addValue(RJ.NODE, reportJob.node)
    params.addValue(RJ.TOTAL_WORK_ITEMS, reportJob.totalWorkItems)
    params.addValue(RJ.COMPLETED_WORK_ITEMS, reportJob.completedWorkItems)
    params.addValue(RJ.SUBMIT_TIME, reportJob.submitTime)
    params.addValue(RJ.START_TIME, reportJob.startTime)
    params.addValue(RJ.END_TIME, reportJob.endTime)
    params.addValue(RJ.USERNAME, reportJob.username)
    params.addValue(RJ.RESULT_URI, reportJob.resultUri)
    params.addValue(RJ.REPORT_DEFINITION, new SqlBinaryValue(reportJob.getReportDefinition.toJson.getBytes(StandardCharsets.UTF_8)), BLOB_TYPE)

    sqlInsert(pkName(RJ.JOB_ID), STMT_CREATE_REPORT_JOB, params, (id: CiUid) => {
      reportJob.setJobId(id)
      reportJob
    })
  }

  private val QUERY_REPORT_JOB_BY_JOB_ID =
    s"""
       |SELECT ${RJ.JOB_ID}, ${RJ.NODE}, ${RJ.REPORT_TYPE},
       |       ${RJ.REPORT_NAME}, ${RJ.STATUS},
       |       ${RJ.TOTAL_WORK_ITEMS}, ${RJ.COMPLETED_WORK_ITEMS},
       |       ${RJ.SUBMIT_TIME}, ${RJ.START_TIME}, ${RJ.END_TIME},
       |       ${RJ.USERNAME}, ${RJ.RESULT_URI}, ${RJ.REPORT_DEFINITION}
       | FROM ${RJ.TABLE}
       | WHERE ${RJ.JOB_ID} = :${RJ.JOB_ID}
       """.stripMargin

  def findByJobId(jobId: ReportJobId): Option[ReportJob] = {
    try {
      Some(namedTemplate.queryForObject(
        QUERY_REPORT_JOB_BY_JOB_ID,
        Map[String, Any](RJ.JOB_ID -> jobId).asJava,
        reportJobMapper
      ))
    } catch {
      case _: EmptyResultDataAccessException => None
    }
  }

  private val STMT_UPDATE_REPORT_JOB =
    s"""UPDATE ${RJ.TABLE}
       | SET
       |  ${RJ.STATUS} = :${RJ.STATUS},
       |  ${RJ.NODE} = :${RJ.NODE},
       |  ${RJ.TOTAL_WORK_ITEMS} = :${RJ.TOTAL_WORK_ITEMS},
       |  ${RJ.COMPLETED_WORK_ITEMS} = :${RJ.COMPLETED_WORK_ITEMS},
       |  ${RJ.SUBMIT_TIME} = :${RJ.SUBMIT_TIME},
       |  ${RJ.START_TIME} = :${RJ.START_TIME},
       |  ${RJ.END_TIME} = :${RJ.END_TIME},
       |  ${RJ.USERNAME} = :${RJ.USERNAME},
       |  ${RJ.RESULT_URI} = :${RJ.RESULT_URI}
       | WHERE ${RJ.JOB_ID} = :${RJ.JOB_ID}
       """.stripMargin

  def update(reportJob: ReportJob): ReportJob = {
    logger.debug(s"Updating report job: $reportJob")
    sqlUpdate(STMT_UPDATE_REPORT_JOB,
      params(
        RJ.JOB_ID -> reportJob.getJobId(),
        RJ.STATUS -> reportJob.status.name(),
        RJ.NODE -> reportJob.node,
        RJ.TOTAL_WORK_ITEMS -> reportJob.totalWorkItems,
        RJ.COMPLETED_WORK_ITEMS -> reportJob.completedWorkItems,
        RJ.SUBMIT_TIME -> reportJob.submitTime,
        RJ.START_TIME -> reportJob.startTime,
        RJ.END_TIME -> reportJob.endTime,
        RJ.USERNAME -> reportJob.username,
        RJ.RESULT_URI -> reportJob.resultUri
      ),
      {
        case 0 => throw new NotFoundException(s"Job [${reportJob.getJobId()}] could not be found.")
        case _ =>
      })
    reportJob
  }

  private val STMT_DELETE_REPORT_JOB = s"DELETE FROM ${RJ.TABLE} WHERE ${RJ.JOB_ID} = :${RJ.JOB_ID}"

  def delete(jobId: ReportJobId): Unit = {
    sqlExec(STMT_DELETE_REPORT_JOB, params(RJ.JOB_ID -> jobId), _.execute())
  }

  override def query(reportJobFilter: ReportJobFilters, pageable: Pageable): Page[ReportJob] = {
    QueryBuilder(dialect, namedTemplate, sqlRepositoryAdapter)
      .from(reportJobFilter)
      .withPageable(pageable)
      .build()
      .execute()
  }

  private val QUERY_STALE_SUBMITTED_JOB_IDS =
    s"SELECT ${RJ.JOB_ID} FROM ${RJ.TABLE} WHERE ${RJ.STATUS} = '${SUBMITTED.name()}' AND ${RJ.SUBMIT_TIME} < :${RJ.SUBMIT_TIME}"

  override def submittedStaleJobIds(olderThan: Date): Seq[ReportJobId] = {
    namedTemplate.queryForList(QUERY_STALE_SUBMITTED_JOB_IDS,
      params(RJ.SUBMIT_TIME -> olderThan).asJava,
      classOf[ReportJobId]
    ).asScala.toSeq
  }

  private val QUERY_STALE_STARTED_JOB_IDS =
    s"SELECT ${RJ.JOB_ID} FROM ${RJ.TABLE} WHERE ${RJ.STATUS} = '${STARTED.name()}' AND ${RJ.START_TIME} < :${RJ.START_TIME}"

  override def startedStaleJobIds(olderThan: Date): Seq[ReportJobId] = {
    namedTemplate.queryForList(QUERY_STALE_STARTED_JOB_IDS,
      params(RJ.START_TIME -> olderThan).asJava,
      classOf[ReportJobId]
    ).asScala.toSeq
  }

  private val QUERY_NODE_SUBMITTED_JOB_IDS =
    s"SELECT ${RJ.JOB_ID} FROM ${RJ.TABLE} WHERE ${RJ.STATUS} = '${SUBMITTED.name()}' AND ${RJ.NODE} = :${RJ.NODE}"

  override def submittedNodeJobIds(node: String): Seq[ReportJobId] = {
    namedTemplate.queryForList(QUERY_NODE_SUBMITTED_JOB_IDS,
      params(RJ.NODE -> node).asJava,
      classOf[ReportJobId]
    ).asScala.toSeq
  }


  private val QUERY_NODE_STARTED_JOB_IDS =
    s"SELECT ${RJ.JOB_ID} FROM ${RJ.TABLE} WHERE ${RJ.STATUS} = '${STARTED.name()}' AND ${RJ.NODE} = :${RJ.NODE}"

  override def startedNodeJobIds(node: String): Seq[ReportJobId] = {
    namedTemplate.queryForList(QUERY_NODE_STARTED_JOB_IDS,
      params(RJ.NODE -> node).asJava,
      classOf[ReportJobId]
    ).asScala.toSeq
  }

  private val STMT_UPDATE_REPORT_JOB_PROGRESS =
    s"""UPDATE ${RJ.TABLE}
       | SET
       |  ${RJ.TOTAL_WORK_ITEMS} = :${RJ.TOTAL_WORK_ITEMS},
       |  ${RJ.COMPLETED_WORK_ITEMS} = :${RJ.COMPLETED_WORK_ITEMS}
       | WHERE ${RJ.JOB_ID} = :${RJ.JOB_ID}
       """.stripMargin

  override def updateProgress(reportJob: ReportJob): ReportJob = {
    logger.debug(s"Updating report job '${reportJob.getJobId()}' progress: ${reportJob.completedWorkItems}/${reportJob.totalWorkItems}")
    sqlUpdate(STMT_UPDATE_REPORT_JOB_PROGRESS,
      params(
        RJ.JOB_ID -> reportJob.getJobId(),
        RJ.TOTAL_WORK_ITEMS -> reportJob.totalWorkItems,
        RJ.COMPLETED_WORK_ITEMS -> reportJob.completedWorkItems
      ),
      {
        case 0 => logger.warn(s"Job [${reportJob.getJobId()}] could not be found.")
        case _ =>
      })
    reportJob

  }

  private val QUERY_STALE_STARTED_JOBS_FROM_OTHER_NODES_IDS =
    s"SELECT ${RJ.JOB_ID} FROM ${RJ.TABLE} WHERE ${RJ.STATUS} = '${STARTED.name()}' AND ${RJ.NODE} NOT IN (:currentNodes)"

  override def startedStaleNodeJobIds(currentNodeIds: Seq[String]): Seq[ReportJobId] = {
    namedTemplate.queryForList(QUERY_STALE_STARTED_JOBS_FROM_OTHER_NODES_IDS,
      params("currentNodes" -> currentNodeIds.asJava).asJava,
      classOf[ReportJobId]
    ).asScala.toSeq
  }

  private val QUERY_STALE_SUBMITTED_JOBS_FROM_OTHER_NODES_IDS =
    s"SELECT ${RJ.JOB_ID} FROM ${RJ.TABLE} WHERE ${RJ.STATUS} = '${SUBMITTED.name()}' AND ${RJ.NODE} NOT IN (:currentNodes)"

  override def submittedStaleNodeJobIds(currentNodeIds: Seq[String]): Seq[ReportJobId] = {
    namedTemplate.queryForList(QUERY_STALE_SUBMITTED_JOBS_FROM_OTHER_NODES_IDS,
      params("currentNodes" -> currentNodeIds.asJava).asJava,
      classOf[ReportJobId]
    ).asScala.toSeq
  }
}

object SqlReportJobRepository {

  implicit class ReportDefinitionExtensions(reportDefinition: ReportDefinition) {
    def toJson: String = CiSerializerHelper.serialize(reportDefinition)
  }

}
