package com.xebialabs.xlrelease.reports.excel

import com.xebialabs.xlrelease.reports.audit.InstanceData
import com.xebialabs.xlrelease.reports.excel.AuditReport.{LOGO_HEIGHT, LOGO_WIDTH}
import com.xebialabs.xlrelease.reports.excel.MultiReleaseAuditReport.FilterData
import com.xebialabs.xlrelease.reports.excel.PermissionReport.{Data, NUMBER_OF_COLUMNS, REPORT_TITLE}
import com.xebialabs.xlrelease.reports.repository.{FolderUserPermissionsReportData, GlobalUserPermissionsReportData}
import org.apache.poi.ss.usermodel.Workbook

import java.util.Date

object PermissionReport {

  val LOGO_WIDTH: Integer = 96 // found by experiment, magic units
  val LOGO_HEIGHT: Integer = 9

  val REPORT_TITLE = "Release User Permissions Report"
  val NUMBER_OF_COLUMNS = 8

  case class Data(username: String,
                  creationDate: Date,
                  instanceData: InstanceData,
                  globalPermissionsData: Iterable[GlobalUserPermissionsReportData],
                  folderPermissionsData: Iterable[FolderUserPermissionsReportData])

  def getWorkBook(data: Data, filters: Option[Seq[FilterData]]): Workbook = {
    new PermissionReport(data, filters).getWorkBook
  }

}

private class PermissionReport(data: Data, filters: Option[Seq[FilterData]]) extends ExcelCommonHeader {

  val workbook: ReportWorkbook = new ReportWorkbook
  var sheetWriter: ExcelSheetWriter = _
  val styles = workbook.getStyles

  def getWorkBook: Workbook = {
    generateTitleWorksheet
    generateGlobalWorksheet
    generateFolderWorksheet

    workbook.getWorkbookWithoutErrorSheet
  }

  private def generateTitleWorksheet: Unit = {
    sheetWriter = workbook.createReportSection("Report info")

    addCommonHeader(data.username, data.creationDate, data.instanceData,
      NUMBER_OF_COLUMNS, LOGO_WIDTH, LOGO_HEIGHT,
      ReportInfo(reportTitle = REPORT_TITLE))

    addNoBorderEmptyRow(NUMBER_OF_COLUMNS)

    if (filters.isDefined && !filters.get.isEmpty) {

      sheetWriter
        .newRow()
        .addCell("Filters", styles.noGridBoldText)
      filters.get.zipWithIndex.foreach {
        case (filterData, filterIndex) => {
          if (filterIndex > 0) {
            sheetWriter.newRow().addCell("")
          }
          sheetWriter.addCell(filterData.label)
          filterData.values.zipWithIndex.foreach {
            case (value, valueIndex) => {
              if (valueIndex > 0) {
                sheetWriter.newRow().addCell("").addCell("")
              }
              sheetWriter.addCell(value)
              fillCellsUntilEndOfRow(NUMBER_OF_COLUMNS, styles.noGrid)
            }
          }
          if (filterData.values.isEmpty) {
            fillCellsUntilEndOfRow(NUMBER_OF_COLUMNS, styles.noGrid)
          }
        }
      }

    }

    addNoBorderEmptyRow(NUMBER_OF_COLUMNS)
    addNoBorderEmptyRow(NUMBER_OF_COLUMNS)
    addNoBorderEmptyRow(NUMBER_OF_COLUMNS)

    sheetWriter.newRow()
    sheetWriter.addHeaderCell("", 30, styles.noGrid)
    sheetWriter.addHeaderCell("", 30, styles.noGrid)
    sheetWriter.addHeaderCell("", 30, styles.noGrid)
    sheetWriter.addHeaderCell("", 30, styles.noGrid)
    sheetWriter.addHeaderCell("", 3, styles.noGrid)
    sheetWriter.addHeaderCell("", 30, styles.noGrid)
    sheetWriter.addHeaderCell("", 8, styles.noGrid)
    sheetWriter.addHeaderCell("", 13, styles.noGrid)
  }

  private def generateGlobalWorksheet: Unit = {

    sheetWriter = workbook.createReportSection("Global permissions")

    sheetWriter.newRow()
    sheetWriter.addHeaderCell(ExcelHeaderColumn("Principal", 30), styles.whiteOnGreen)
    sheetWriter.addHeaderCell(ExcelHeaderColumn("Full Name", 30), styles.whiteOnGreen)
    sheetWriter.addHeaderCell(ExcelHeaderColumn("Has global permission", 30), styles.whiteOnGreen)
    sheetWriter.addHeaderCell(ExcelHeaderColumn("Because of role", 30), styles.whiteOnGreen)

    data.globalPermissionsData
      .toList
      .sortBy(containerPermissionAndPrincipal => (containerPermissionAndPrincipal.principal.toLowerCase, containerPermissionAndPrincipal.permission.toLowerCase))
      .foreach {
        case GlobalUserPermissionsReportData(principal, userFullName, permission, roles) =>
          sheetWriter.newRow()
          sheetWriter.addCell(principal, styles.default)
          sheetWriter.addCell(userFullName, styles.default)
          sheetWriter.addCell(permission, styles.default)
          sheetWriter.addCell(roles.mkString(", "), styles.default)
      }
  }

  private def generateFolderWorksheet: Unit = {

    sheetWriter = workbook.createReportSection("Folder permissions")

    sheetWriter.newRow()
    sheetWriter.addHeaderCell(ExcelHeaderColumn("Principal", 30), styles.whiteOnGreen)
    sheetWriter.addHeaderCell(ExcelHeaderColumn("Full Name", 30), styles.whiteOnGreen)
    sheetWriter.addHeaderCell(ExcelHeaderColumn("In folder", 30), styles.whiteOnGreen)
    sheetWriter.addHeaderCell(ExcelHeaderColumn("Has permission", 30), styles.whiteOnGreen)
    sheetWriter.addHeaderCell(ExcelHeaderColumn("Because of team (global role)", 30), styles.whiteOnGreen)

    data.folderPermissionsData
      .toList
      .sortBy(containerPermissionAndPrincipal => (containerPermissionAndPrincipal.principal.toLowerCase, containerPermissionAndPrincipal.folderName.toLowerCase, containerPermissionAndPrincipal.permission.toLowerCase))
      .toIterable
      .foreach {
        case FolderUserPermissionsReportData(folderName, principal, userFullName, permission, links) =>
          sheetWriter.newRow()
          sheetWriter.addCell(principal, styles.default)
          sheetWriter.addCell(userFullName, styles.default)
          sheetWriter.addCell(folderName, styles.default);
          sheetWriter.addCell(permission, styles.default)
          sheetWriter.addCell(links.mkString(", "), styles.default)
      }
  }

}
