package com.xebialabs.xlrelease.reports.domain

import com.xebialabs.xlrelease.reports.domain.exceptions.{MultipleExceptions, NullValueException}
import grizzled.slf4j.Logging

import scala.util.{Failure, Success, Try}

object MaybeData extends Logging {

  type Maybe[R] = Try[Either[(Throwable, R), R]]

  def failure[R](t: Throwable) : Maybe[R] = {
    logger.trace(s"Got exception while wrapping data into Maybe.", t)
    Failure(t)
  }
  def partial[R](t: Throwable, value: R) = Success(Left(t -> value))
  def partial[R](t: (Throwable, R)) = Success(Left(t))
  def success[R](value: R) = Success(Right(value))

  def fromTry[R](t: Try[R]): Maybe[R] =
    t.map(Right.apply).recoverWith { case t: Throwable => failure(t) }

  def allowingNull[R](value: => R): Maybe[R] = fromTry(Try(value))

  def apply[R](value: => R): Maybe[R] = {
    allowingNull(value).map(_.flatMap {
      case null => Left(NullValueException() -> value)
      case other => Right(other)
    })
  }

  def nonEmptyString(value: => String): Maybe[String] = {
    apply(value).map(_.flatMap {
      case s if s.isEmpty => Left(NullValueException() -> s)
      case nonEmpty => Right(nonEmpty)
    })
  }

  def recoverWith[R](t: Try[R], default: Throwable => (Throwable, R)): Maybe[R] = {
    recoverMaybe(fromTry(t), default)
  }

  def recoverMaybe[R](t: Maybe[R], default: Throwable => (Throwable, R)): Maybe[R] = {
    t.recover {
      case e: Throwable =>
        Left(default(e))
    }
  }

  def combine[A, B, C](ma: Maybe[A], mb: Maybe[B])(f: (A, B) => C): Maybe[C] = {

    ma.flatMapMaybe { a =>
      mb.mapValue { b =>
        f(a, b)
      }
    }
  }

  implicit class EitherOps[A](val _either: Either[(Throwable, A), A]) extends AnyVal {
    def getValue: A = _either.fold(_._2, identity)
  }

  implicit class FlattenMaybeOps[A](val _maybe: Maybe[Maybe[A]]) {
    def flattenMaybe(): Maybe[A] = _maybe.flatMap {
      case Right(inner) => inner
      case Left((t0, inner)) =>
        inner.map {
          case Right(v) => Left(t0 -> v)
          case Left((t1, v)) => Left(MultipleExceptions(t0, t1) -> v)
        }
    }
  }

  implicit class MaybeOps[A](val _maybe: Maybe[A]) extends AnyVal {
    def getValue: A = _maybe.map(_.getValue).get

    def mapValue[B](f: A => B): Maybe[B] =
      _maybe.map {
        case Left((t, v)) => Left(t -> f(v))
        case Right(v) => Right(f(v))
      }

    def flatMapMaybe[B](fn: A => Maybe[B]): Maybe[B] = {
      _maybe.mapValue(fn).flattenMaybe()
    }

    def mapMaybe[B](fn: A => B): Maybe[B] = flatMapMaybe(a => MaybeData(fn(a)))

    def hasType[T](theType: Class[T]): Boolean = _maybe match {
      case Success(success) => success match {
        case Left((_, value)) => if (value.isInstanceOf[T]) true else false
        case Right(value) => if (value.isInstanceOf[T]) true else false
      }
      case _ => false
    }
  }
}
