package com.xebialabs.xlrelease.reports.job.impl

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName
import com.xebialabs.deployit.security.{PermissionDeniedException, PermissionEnforcer}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.reports.job.api.ReportingEngineService.ReportJobId
import com.xebialabs.xlrelease.reports.job.api.{ReportDefinition, ReportingEngineService, StreamingReportResult}
import com.xebialabs.xlrelease.reports.job.domain.{ReportJob, ReportJobStatus}
import com.xebialabs.xlrelease.reports.job.events._
import com.xebialabs.xlrelease.reports.job.repository.{ReportJobFilters, ReportJobRepository}
import com.xebialabs.xlrelease.service.BroadcastService
import grizzled.slf4j.Logging
import org.springframework.context.ApplicationContext
import org.springframework.data.domain.{Page, PageRequest, Pageable}

import java.util.{List => JList}
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

class DefaultReportingEngineService(applicationContext: ApplicationContext,
                                    reportJobInstanceFactory: ReportJobInstanceFactory,
                                    reportJobRepository: ReportJobRepository,
                                    globalReportStorage: GlobalReportStorage,
                                    permissionEnforcer: PermissionEnforcer,
                                    eventBus: XLReleaseEventBus,
                                    broadcastService: BroadcastService
                                   )
  extends ReportingEngineService with Logging {

  override def status(jobId: ReportJobId): ReportJobStatus = {
    reportJobRepository.findByJobId(jobId).map(reportJob => {
      checkReportOwner(reportJob)
      reportJob.getStatus
    }).getOrElse(throw new NotFoundException(s"Job '$jobId' could not be found."))
  }

  override def getResult(jobId: ReportJobId): StreamingReportResult = {
    val reportJobOption = reportJobRepository.findByJobId(jobId)
    reportJobOption.map(reportJob => {
      checkReportOwner(reportJob)
      globalReportStorage.resolve(globalReportStorage.reportResult(reportJob))
    }).getOrElse(throw new NotFoundException(s"Result of job '$jobId' could not be found."))
  }

  override def preview(reportDefinition: ReportDefinition): Any = {
    applicationContext.getAutowireCapableBeanFactory.autowireBean(reportDefinition)
    reportDefinition.preview()
  }

  override def get(jobId: ReportJobId): ReportJob = {
    val reportJobOption = reportJobRepository.findByJobId(jobId)
    reportJobOption.map(reportJob => {
      checkReportOwner(reportJob)
      reportJob
    }).getOrElse(throw new NotFoundException(s"Job '$jobId' could not be found."))
  }

  override def submit(reportDefinition: ReportDefinition): ReportJobId = {
    val reportJobInstance = reportJobInstanceFactory.submit(reportDefinition)
    reportJobInstance.enqueue()
    val jobId = reportJobInstance.getJobId
    jobId
  }

  override def abort(jobId: ReportJobId): Unit = {
    val reportJobOption = reportJobRepository.findByJobId(jobId)
    reportJobOption
      .map(reportJob => abort(reportJob))
      .getOrElse(throw new NotFoundException(s"Job '$jobId' could not be found."))
  }

  override def abort(jobIds: JList[ReportJobId]): JList[ReportJobId] = {
    if (jobIds.isEmpty) {
      Seq.empty.asJava
    } else {
      val reportJobFilters = new ReportJobFilters
      reportJobFilters.setJobIds(jobIds)
      findAndAbortReportJobs(reportJobFilters)
    }
  }

  override def abortAll(): JList[ReportJobId] = {
    val reportJobFilters = new ReportJobFilters
    findAndAbortReportJobs(reportJobFilters)
  }

  override def delete(jobId: ReportJobId): Unit = {
    val reportJobOption = reportJobRepository.findByJobId(jobId)
    reportJobOption
      .map(reportJob => deleteReportJob(reportJob))
      .getOrElse(throw new NotFoundException(s"Job '$jobId' could not be found."))
  }

  override def delete(jobIds: JList[ReportJobId]): JList[ReportJobId] = {
    if (jobIds.isEmpty) {
      Seq.empty.asJava
    } else {
      val reportJobFilters = new ReportJobFilters
      reportJobFilters.setJobIds(jobIds)
      findAndDeleteReportJobs(reportJobFilters)
    }
  }

  override def deleteAll(): JList[ReportJobId] = {
    val reportJobFilters = new ReportJobFilters
    reportJobFilters.statuses = List(ReportJobStatus.ABORTED.name(), ReportJobStatus.FAILED.name(), ReportJobStatus.COMPLETED.name()).asJava
    findAndDeleteReportJobs(reportJobFilters)
  }

  override def findBy(reportJobFilters: ReportJobFilters, pageable: Pageable): Page[ReportJob] = {
    if (!permissionEnforcer.isCurrentUserAdmin) {
      reportJobFilters.username = getAuthenticatedUserName
    }
    reportJobRepository.query(reportJobFilters, pageable)
  }

  private def findAndAbortReportJobs(reportJobFilters: ReportJobFilters): JList[ReportJobId] = {
    val abortRequestedJobIds = ListBuffer.empty[ReportJobId]
    reportJobFilters.statuses = List(ReportJobStatus.SUBMITTED.name(), ReportJobStatus.STARTED.name()).asJava
    val result = findBy(reportJobFilters, PageRequest.of(0, Int.MaxValue))
    result.forEach {
      reportJob => {
        try {
          abortRequestedJobIds += abort(reportJob)
        } catch {
          case e: Exception => logger.warn("Error while aborting a list of jobs", e)
        }
      }
    }
    abortRequestedJobIds.asJava
  }

  private def abort(reportJob: ReportJob): ReportJobId = {
    checkReportOwner(reportJob)
    val jobId = reportJob.getJobId()
    val validStatuses = List(ReportJobStatus.SUBMITTED.name(), ReportJobStatus.STARTED.name())
    if (validStatuses.contains(reportJob.status.name())) {
      broadcastService.broadcast(AbortReportJob(jobId), publishEventOnSelf = true)
      jobId
    } else {
      throw new IllegalStateException(s"Can not abort report job '${jobId}'. " +
        "It is already completed, failed or aborted.")
    }
  }

  private def findAndDeleteReportJobs(reportJobFilters: ReportJobFilters): JList[ReportJobId] = {
    val deletedJobIds = ListBuffer.empty[ReportJobId]
    val result = findBy(reportJobFilters, PageRequest.of(0, Int.MaxValue))
    result.forEach {
      reportJob => {
        try {
          deletedJobIds += deleteReportJob(reportJob).getJobId()
        } catch {
          case e: Exception => logger.warn("Error while deleting a list of jobs", e)
        }
      }
    }
    deletedJobIds.asJava
  }

  private def deleteReportJob(reportJob: ReportJob): ReportJob = {
    checkReportOwner(reportJob)
    val validStatuses = List(ReportJobStatus.ABORTED.name(), ReportJobStatus.FAILED.name(), ReportJobStatus.COMPLETED.name())
    if (validStatuses.contains(reportJob.status.name()) && globalReportStorage.cleanup(reportJob)) {
      reportJob.status = ReportJobStatus.DELETED
      val deletedReportJob = reportJobRepository.update(reportJob)
      eventBus.publish(ReportJobDeletedEvent(deletedReportJob))
      deletedReportJob
    } else {
      throw new IllegalStateException(s"Can not delete report job '${reportJob.getJobId}'. " +
        s"It is either already deleted or Digital.ai Release does not have required access to the filesystem.")
    }
  }

  private def checkReportOwner(reportJob: ReportJob): Unit = {
    if (!permissionEnforcer.isCurrentUserAdmin && reportJob.username != getAuthenticatedUserName) {
      throw PermissionDeniedException.withMessage(s"You can not take action on report job '${reportJob.getJobId}' as you are not the owner of this report job.")
    }
  }

}
