package com.xebialabs.xlrelease.reports.job.impl.filesystem

import com.xebialabs.xlrelease.reports.job.api.ReportingEngineService.ReportJobId
import com.xebialabs.xlrelease.reports.job.api.{ReportResult, ReportStorage, StreamingReportResult}
import com.xebialabs.xlrelease.reports.job.domain.ReportJob
import com.xebialabs.xlrelease.storage.local.LocalStorageConfig
import grizzled.slf4j.Logging
import org.apache.commons.io.FileUtils

import java.io.File
import java.text.SimpleDateFormat
import java.util.Date
import scala.annotation.tailrec

class FileSystemReportStorage(localStorageConfig: LocalStorageConfig) extends ReportStorage with Logging {
  private[impl] val rootReportDir: File = localStorageConfig.basePath.toFile

  if (!rootReportDir.exists) {
    try {
      rootReportDir.mkdirs()
      logger.debug(s"Created root directory for reports at '${rootReportDir.getAbsolutePath}' based on provided location '${localStorageConfig.basePath}'")
    } catch {
      case e: Exception =>
        val msg = s"Unable to create root directory for reports '${localStorageConfig.basePath}'."
        throw new IllegalStateException(msg, e)
    }
  }

  if (!rootReportDir.isDirectory) {
    val msg = s"Report storage root directory location '${localStorageConfig.basePath}' does not point to a directory."
    throw new IllegalStateException(msg)
  }

  override def reportStorage(jobId: ReportJobId, generatedOn: Date): FileSystemStreamingReportStorage = {
    val reportWorkdir: File = getReportDirectory(jobId, generatedOn)
    if (!reportWorkdir.exists) {
      val isCreated = try {
        reportWorkdir.mkdirs()
      } catch {
        case e: Exception =>
          val msg = s"Unable to create directory for report job '$jobId' in '$rootReportDir'."
          throw new IllegalStateException(msg, e)
      }
      if (!isCreated) {
        throw new IllegalStateException(s"Unable to create directory for report job '$jobId' in '$rootReportDir'.")
      }
    }
    logger.debug(s"Created root directory for reports at '${reportWorkdir.getAbsolutePath}' based on provided location '$rootReportDir'")
    // for now work and report are same, work could be different in case if for report we use network storage
    new FileSystemStreamingReportStorage(reportWorkdir, reportWorkdir)
  }

  override def resolve(reportResult: ReportResult): StreamingReportResult = {
    reportResult match {
      case streamingReportResult: StreamingReportResult => streamingReportResult
      case _ =>
        val jobStorage = reportStorage(reportResult.jobId, reportResult.generatedOn)
        val jobDir = jobStorage.reportDir
        if (!jobDir.exists()) {
          throw new IllegalStateException(s"Report directory of job '${reportResult.jobId}' does not exist.")
        }
        new FileSystemStreamingReportResult(jobDir.toURI, reportResult)
    }
  }

  def cleanup(reportJob: ReportJob): Boolean = {
    try {
      val jobDir = getReportDirectory(reportJob.getJobId(), reportJob.getEndTime)

      if (jobDir.exists()) {
        logger.debug(s"Deleting directory: ${jobDir.getPath}")
        FileUtils.deleteDirectory(jobDir)
      }

      try {
        deleteParentsIfEmpty(jobDir)
      } catch {
        case t: Throwable =>
          logger.warn(s"Exception while deleting report's parent directory: ${t.toString}", t)
      }

      true
    } catch {
      case e: Exception =>
        logger.warn(s"Exception while deleting report's job directory: ${e.getMessage}")
        false
    }
  }

  private[impl] def getReportDirectory(jobId: ReportJobId, generatedOn: Date): File = {
    val formatter = new SimpleDateFormat("yyyy/MM/dd")
    val dateFragment = formatter.format(generatedOn)
    val reportWorkdir: File = new File(rootReportDir, s"$dateFragment/$jobId")
    reportWorkdir
  }

  @tailrec
  private final def deleteParentsIfEmpty(dir: File): Unit = {
    val parent = dir.getParentFile
    if (!parent.equals(rootReportDir) && parent.isDirectory && parent.list().isEmpty) {
      logger.trace(s"Deleting directory: ${parent.getPath}")
      FileUtils.deleteDirectory(parent)
      deleteParentsIfEmpty(parent)
    }
  }
}
