package com.xebialabs.xlrelease.reports.repository.build

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters
import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.udm.reporting.BuildRecord
import com.xebialabs.xlrelease.domain.utils.FullReleaseId
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.reports.repository.CommonRowAndParamsMapper._
import com.xebialabs.xlrelease.reports.repository.build.BuildRecordArchiveRepository._
import com.xebialabs.xlrelease.reports.repository.{ArchivedTasksRepository, RepositoryExceptionUtils, SearchRecordsSupport}
import com.xebialabs.xlrelease.repository.FacetRepository
import com.xebialabs.xlrelease.repository.FacetRepository.SpecializedFacetRepository
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.collection.mutable
import scala.util.Failure


object BuildRecordArchiveRepository {

  import BUILD_TASK_REPORTING_RECORD_ARCHIVE._

  private lazy val ALL_FIELDS =
    s"""| ${commonFieldNames(BUILD_TASK_REPORTING_RECORD_ARCHIVE, "b")},
        | t.$REPORT_TASKS_ID_COLUMN,
        | b.$PROJECT,
        | b.$BUILD,
        | b.$OUTCOME,
        | b.$START_DATE,
        | b.$END_DATE,
        | b.$DURATION,
        | b.$BUILD_URL""".stripMargin

  private lazy val SELECT_ALL_BUILD_RECORDS =
    s"""SELECT
       | $ALL_FIELDS
       |FROM $TABLE b
       |JOIN $REPORT_TASKS_TABLE_NAME t ON t.$REPORT_TASKS_UID_COLUMN = b.$TASK_UID""".stripMargin

  private lazy val SELECT_BUILD_RECORD_BY_ID_STMT: String =
    s"""|$SELECT_ALL_BUILD_RECORDS
        |WHERE b.$RECORD_ID = :recordId""".stripMargin

  private lazy val SELECT_BUILD_RECORDS_BY_TASK_ID_STMT =
    s"""|$SELECT_ALL_BUILD_RECORDS
        |WHERE t.$REPORT_TASKS_ID_COLUMN = :taskId""".stripMargin

  private lazy val SELECT_BUILD_RECORDS_BY_RELEASE_ID_STMT: String =
    s"""|$SELECT_ALL_BUILD_RECORDS
        |JOIN $REPORT_RELEASES_TABLE_NAME r ON r.$REPORT_RELEASES_ID_COLUMN = t.$REPORT_TASKS_RELEASEID_COLUMN
        |WHERE r.$REPORT_RELEASES_ID_COLUMN = :releaseId AND r.$REPORT_RELEASES_PRE_ARCHIVED <> 1""".stripMargin

  private lazy val EXISTS_BUILD_RECORD_BY_ID: String =
    s"""SELECT COUNT($RECORD_ID) FROM $TABLE WHERE $RECORD_ID = :recordId""".stripMargin

  private lazy val INSERT_BUILD_RECORD_STMT =
    s"""| INSERT INTO $TABLE (
       | ${commonFieldNames(BUILD_TASK_REPORTING_RECORD_ARCHIVE)},
       | $TASK_UID,
       | $PROJECT,
       | $BUILD,
       | $OUTCOME,
       | $START_DATE,
       | $END_DATE,
       | $DURATION,
       | $BUILD_URL
       |) VALUES (
       | ${commonParameterNames()},
       | :taskUid,
       | :project,
       | :build,
       | :outcome,
       | :startDate,
       | :endDate,
       | :duration,
       | :buildUrl
       |)""".stripMargin

  private lazy val DELETE_BUILD_RECORD_STMT: String =
    s"""|DELETE FROM $TABLE
        |WHERE $RECORD_ID = :recordId""".stripMargin
}

class BuildRecordArchiveRepository(archivedTasksRepository: ArchivedTasksRepository,
                                   val ciIdService: CiIdService,
                                   val jdbcTemplate: JdbcTemplate,
                                   val dialect: Dialect)
  extends SpecializedFacetRepository(Type.valueOf(classOf[BuildRecord]))
    with FacetRepository.ForArchive
    with PersistenceSupport
    with SearchRecordsSupport[BuildRecord]
    with RepositoryExceptionUtils
    with Logging {

  override def get(recordId: String, recordType: Option[Type]): Facet = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(SELECT_BUILD_RECORD_BY_ID_STMT, params("recordId" -> getName(recordId.normalized)), buildRowMapper)
      .headOption.getOrElse(throw new NotFoundException(s"Record '$recordId' not found."))
  }

  override protected def doCreate(facet: Facet): Facet = {
    logger.debug("Creating Build record in archive")
    requiresInstanceOfSupportedType(facet.getType)
    archivedTasksRepository.taskUidById(shortenId(facet.getTargetId)).map { taskUid =>
      val buildRecord = facet.asInstanceOf[BuildRecord]
      sqlInsert(INSERT_BUILD_RECORD_STMT, asQueryParameters(buildRecord, taskUid))
      facet
    }.recoverWith {
      case _: NotFoundException =>
        Failure(targetTaskNotFoundException(facet.getId, facet.getTargetId))
    }.get
  }

  override def delete(recordId: String, facetType: Option[Type]): Unit =
    sqlUpdate(DELETE_BUILD_RECORD_STMT, params("recordId" -> getName(recordId.normalized)), _ => ())

  override def update(facet: Facet): Facet = throw new UnsupportedOperationException("You cannot update records!")

  override def search(facetsFilters: FacetFilters): Seq[Facet] = searchRecords(facetsFilters, SELECT_ALL_BUILD_RECORDS, buildRowMapper)

  override def exists(recordId: String, recordType: Option[Type]): Boolean = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(EXISTS_BUILD_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), _.getInt(1) > 0).headOption.exists(identity)
  }

  override def findAllFacetsByRelease(release: Release): Seq[Facet] = findAllFacetsByReleaseId(release.getId).toSeq

  override def findAllFacetsByTask(task: Task): Seq[Facet] = findAllFacetsByTargetId(shortenId(task.getId)).toSeq

  override protected def findAllFacetsByTargetId(targetId: CiId): mutable.Seq[BuildRecord] =
    sqlQuery(SELECT_BUILD_RECORDS_BY_TASK_ID_STMT, params("taskId" -> targetId), buildRowMapper)

  override protected def findAllFacetsByReleaseId(releaseId: CiId): mutable.Seq[BuildRecord] =
    sqlQuery[BuildRecord](SELECT_BUILD_RECORDS_BY_RELEASE_ID_STMT, params(
      "releaseId" -> FullReleaseId(releaseId).withOnlyOneParentOrApplicationsForArchiveDb()
    ), buildRowMapper)

  private val buildRowMapper: RowMapper[BuildRecord] = (rs: ResultSet, _: Int) => {
    val buildFacet: BuildRecord = Type.valueOf(rs.getString(BUILD_TASK_REPORTING_RECORD_ARCHIVE.RECORD_TYPE))
      .getDescriptor.newInstance[BuildRecord](rs.getString(BUILD_TASK_REPORTING_RECORD_ARCHIVE.RECORD_ID))
    buildItemRowMapper(rs, buildFacet, rs.getString(REPORT_TASKS_ID_COLUMN), BUILD_TASK_REPORTING_RECORD_ARCHIVE)
  }

}
