package com.xebialabs.xlrelease.reports.repository.build

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.udm.reporting.BuildRecord
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.reports.repository.CommonRowAndParamsMapper._
import com.xebialabs.xlrelease.reports.repository.build.BuildRecordLiveRepository._
import com.xebialabs.xlrelease.reports.repository.{RepositoryExceptionUtils, SearchRecordsSupport}
import com.xebialabs.xlrelease.repository.FacetRepository.SpecializedFacetRepository
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.{CiId, _}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, RELEASES, TASKS}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, TaskCiUid, TaskPersistence}
import com.xebialabs.xlrelease.repository.{FacetRepository, Ids}
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.utils.FolderId
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.collection.mutable

object BuildRecordLiveRepository {

  import BUILD_TASK_REPORTING_RECORD_LIVE._

  private lazy val ALL_FIELDS =
    s"""|${commonFieldNames(BUILD_TASK_REPORTING_RECORD_LIVE, "b")},
       |b.$PROJECT,
       |b.$BUILD,
       |b.$OUTCOME,
       |b.$START_DATE,
       |b.$END_DATE,
       |b.$DURATION,
       |b.$BUILD_URL,
       |f.${FOLDERS.FOLDER_PATH},
       |f.${FOLDERS.FOLDER_ID},
       |r.${RELEASES.RELEASE_ID},
       |t.${TASKS.TASK_ID}""".stripMargin

  private lazy val SELECT_ALL_BUILD_RECORDS: String =
    s"""|SELECT
        | $ALL_FIELDS
        |FROM $TABLE b
        |JOIN ${TASKS.TABLE} t ON t.${TASKS.CI_UID} = b.$TASK_UID
        |JOIN ${RELEASES.TABLE} r ON r.${RELEASES.CI_UID} = t.${TASKS.RELEASE_UID}
        |LEFT OUTER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = r.${RELEASES.FOLDER_CI_UID}""".stripMargin

  private lazy val SELECT_BUILD_RECORD_BY_ID_STMT: String =
    s"""|$SELECT_ALL_BUILD_RECORDS
        |WHERE b.$RECORD_ID = :recordId""".stripMargin

  private lazy val SELECT_BUILD_RECORDS_BY_TASK_ID_STMT: String =
    s"""|$SELECT_ALL_BUILD_RECORDS
        |WHERE t.${TASKS.TASK_ID} = :taskId""".stripMargin

  private lazy val SELECT_BUILD_RECORDS_BY_RELEASE_ID_STMT: String =
    s"""|$SELECT_ALL_BUILD_RECORDS
        |WHERE r.${RELEASES.RELEASE_ID} = :releaseId""".stripMargin

  private lazy val EXISTS_BUILD_RECORD_BY_ID: String =
    s"""SELECT COUNT($RECORD_ID) FROM $TABLE WHERE $RECORD_ID = :recordId""".stripMargin

  private lazy val INSERT_BUILD_RECORD_STMT: String =
    s"""| INSERT INTO $TABLE (
        | ${commonFieldNames(BUILD_TASK_REPORTING_RECORD_LIVE)},
       | $TASK_UID,
       | $PROJECT,
       | $BUILD,
       | $OUTCOME,
       | $START_DATE,
       | $END_DATE,
       | $DURATION,
       | $BUILD_URL
       |) VALUES (
       | ${commonParameterNames()},
       | :taskUid,
       | :project,
       | :build,
       | :outcome,
       | :startDate,
       | :endDate,
       | :duration,
       | :buildUrl
       |)""".stripMargin
  private lazy val DELETE_BUILD_RECORD_STMT: String =
    s"""|DELETE FROM $TABLE
        |WHERE $RECORD_ID = :recordId""".stripMargin
}

class BuildRecordLiveRepository(taskPersistence: TaskPersistence,
                                val ciIdService: CiIdService,
                                val jdbcTemplate: JdbcTemplate,
                                val dialect: Dialect) extends SpecializedFacetRepository(Type.valueOf(classOf[BuildRecord]))
  with FacetRepository.ForLive
  with PersistenceSupport
  with SearchRecordsSupport[BuildRecord]
  with RepositoryExceptionUtils
  with Logging {

  override def get(recordId: String, facetType: Option[Type]): Facet = {
    facetType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(SELECT_BUILD_RECORD_BY_ID_STMT, params("recordId" -> getName(recordId.normalized)), buildRowMapper)
      .headOption.getOrElse(throw new NotFoundException(s"Facet '$recordId' not found."))
  }

  override protected def doCreate(facet: Facet): Facet = {
    logger.debug("Creating Build record in live")
    requiresInstanceOfSupportedType(facet.getType)
    withTargetTaskUid(facet.getTargetId) { taskUid =>
      val buildRecord = facet.asInstanceOf[BuildRecord]
      sqlInsert(INSERT_BUILD_RECORD_STMT, asQueryParameters(buildRecord, taskUid))
      facet.setTargetId(facet.getTargetId)
      facet
    }.getOrElse {
      throw targetTaskNotFoundException(facet.getId, facet.getTargetId)
    }
  }

  override def delete(recordId: String, recordType: Option[Type]): Unit =
    sqlUpdate(DELETE_BUILD_RECORD_STMT, params("recordId" -> getName(recordId.normalized)), _ => ())

  override def update(facet: Facet): Facet = throw new UnsupportedOperationException("You cannot update this records!")

  override def search(facetsFilters: FacetFilters): Seq[Facet] = searchRecords(facetsFilters, SELECT_ALL_BUILD_RECORDS, buildRowMapper)

  override def exists(recordId: String, recordType: Option[Type]): Boolean = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(EXISTS_BUILD_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), _.getInt(1) > 0).headOption.exists(identity)
  }

  override def findAllFacetsByRelease(release: Release): Seq[Facet] = findAllFacetsByReleaseId(release.getId).toSeq

  override def findAllFacetsByTask(task: Task): Seq[Facet] = findAllFacetsByTargetId(task.getId).toSeq

  override protected def findAllFacetsByTargetId(targetId: CiId): mutable.Seq[BuildRecord] =
    sqlQuery(SELECT_BUILD_RECORDS_BY_TASK_ID_STMT, params("taskId" -> getFolderlessId(targetId)), buildRowMapper)

  override protected def findAllFacetsByReleaseId(releaseId: CiId): mutable.Seq[BuildRecord] =
    sqlQuery(SELECT_BUILD_RECORDS_BY_RELEASE_ID_STMT, params("releaseId" -> getName(releaseId.normalized)), buildRowMapper)

  private def withTargetTaskUid[A](targetId: CiId)(f: TaskCiUid => A): Option[A] =
    taskPersistence.getTaskUidById(targetId).map(f)

  private val buildRowMapper: RowMapper[BuildRecord] = (rs: ResultSet, _: Int) => {
    val buildFacet: BuildRecord = Type.valueOf(rs.getString(BUILD_TASK_REPORTING_RECORD_LIVE.RECORD_TYPE))
      .getDescriptor.newInstance[BuildRecord](rs.getString(BUILD_TASK_REPORTING_RECORD_LIVE.RECORD_ID))
    val folderPath = Option(rs.getString(FOLDERS.FOLDER_PATH)).map(FolderId.apply).getOrElse(FolderId.Root)
    val releaseId = rs.getString(RELEASES.RELEASE_ID)
    val taskId = Ids.getReleaselessChildId(rs.getString(TASKS.TASK_ID))
    val targetId = folderPath / rs.getString(FOLDERS.FOLDER_ID) / releaseId / taskId
    buildItemRowMapper(rs, buildFacet, targetId.absolute, BUILD_TASK_REPORTING_RECORD_LIVE)
  }
}
