package com.xebialabs.xlrelease.reports.job.domain

import com.fasterxml.jackson.databind.annotation.JsonSerialize
import com.xebialabs.deployit.plugin.api.udm.Metadata
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.xlrelease.reports.job.api.ReportDefinition
import com.xebialabs.xlrelease.serialization.json.jackson.CiSerializer
import com.xebialabs.xlrelease.utils.XLTypes._

import java.text.SimpleDateFormat
import java.time.Instant
import java.util
import java.util.Date
import scala.beans.BeanProperty

@JsonSerialize(using = classOf[CiSerializer])
@Metadata(description="Report job", versioned = false)
class ReportJob extends BaseConfigurationItem {

  private[this] var _jobId: Integer = _

  def getJobId(): Integer = _jobId

  def setJobId(value: Integer): Unit = {
    _jobId = value
    this.setId(value.toString)
  }

  @BeanProperty
  @XLProperty(label = "Node", description="Node where job executes", hidden = true, required = false)
  var node: String = _

  @BeanProperty
  @XLProperty(label = "Type", description = "Type of the report job")
  var reportType: String = _

  @BeanProperty
  @XLProperty(label = "Name", description = "Unique name of your report")
  var reportName: String = _

  @BeanProperty
  @XLProperty(label = "Status", description = "Current status of the job")
  @volatile // can be changed from another thread during abort
  var status: ReportJobStatus = _

  @BeanProperty
  @XLProperty(label = "Total", description = "Total number of work items to process", required = false)
  var totalWorkItems: Integer = _

  @BeanProperty
  @XLProperty(label = "Completed", description = "Number of work items processed", required = false, defaultValue = "0")
  var completedWorkItems: Integer = 0

  @BeanProperty
  @XLProperty(label = "Submission time", description = "Time when job was submitted")
  var submitTime: Date = _

  @BeanProperty
  @XLProperty(label = "Start time", description = "Time when job was started", required = false)
  var startTime: Date = _

  @BeanProperty
  @XLProperty(label = "End time", description = "Time when job was completed", required = false)
  var endTime: Date = _

  @BeanProperty
  @XLProperty(label = "User", description = "User who submitted the job")
  var username: String = _

  @BeanProperty
  @XLProperty(label = "Result URI", description = "URI location of the job output", required = false)
  var resultUri: String = _

  @XLProperty(label = "Report definition", description = "Report definition.", asContainment = true)
  private var reportDefinitions: java.util.List[ReportDefinition] = new util.ArrayList[ReportDefinition]()

  def setReportDefinition(reportDefinition: ReportDefinition): Unit = {
    if (this.reportDefinitions.isEmpty) {
      reportDefinitions.add(reportDefinition)
    } else {
      reportDefinitions.set(0, reportDefinition)
    }
  }

  def getReportDefinition: ReportDefinition = {
    if (reportDefinitions.isEmpty) {
      null
    } else {
      this.reportDefinitions.get(0)
    }
  }

  override def toString: String = {
    s"{jobId: ${_jobId}, node: $node, status: $status, totalWorkItems: $totalWorkItems, completedWorkItems: $completedWorkItems}"
  }
}

object ReportJob {

  def apply(reportDefinition: ReportDefinition): ReportJob = {
    val format = new SimpleDateFormat("yyyy-MM-dd_HHmmss")
    val reportJob = new ReportJob()
    reportJob.setReportType(reportDefinition.getType.toString)
    reportJob.setReportName(s"${format.format(reportDefinition.getGeneratedOn())} - ${reportDefinition.reportName}")
    reportJob.setReportDefinition(reportDefinition)
    reportJob.setSubmitTime(currentTime)
    reportJob.setStatus(ReportJobStatus.SUBMITTED)
    reportJob.setUsername(reportDefinition.getGeneratedBy())
    reportJob
  }

  private def currentTime: Date = Date.from(Instant.now())

}
