package com.xebialabs.xlrelease.reports.pdf

import grizzled.slf4j.Logging
import org.apache.pdfbox.pdmodel.PDPageContentStream.AppendMode
import org.apache.pdfbox.pdmodel.common.PDRectangle
import org.apache.pdfbox.pdmodel.graphics.image.PDImageXObject
import org.apache.pdfbox.pdmodel.{PDDocument, PDDocumentInformation, PDPage, PDPageContentStream}
import org.springframework.core.io.DefaultResourceLoader
import org.springframework.stereotype.Service
import org.springframework.util.StreamUtils

import java.io.{BufferedOutputStream, OutputStream}
import java.util.Base64

trait DashboardReportService {

  def generateDashboardReport(pngContent: String, title: String, outputStream: OutputStream): Unit
}

@Service
class PdfDashboardReportService extends DashboardReportService with Logging {


  def generateDashboardReport(pngContent: String, title: String, outputStream: OutputStream): Unit = {
    import PdfDashboardReportService._
    System.setProperty("org.apache.pdfbox.rendering.UsePureJavaCMYKConversion", "true")
    System.setProperty("sun.java2d.cmm", "sun.java2d.cmm.kcms.KcmsServiceProvider")
    val doc = new PDDocument()
    try {
      val page = new PDPage(PDRectangle.A4)
      val docInfo: PDDocumentInformation = new PDDocumentInformation()
      docInfo.setAuthor("Digital.ai Release")
      docInfo.setTitle(title)
      doc.setDocumentInformation(docInfo)
      doc.addPage(page)
      val contentStream = new PDPageContentStream(doc, page, AppendMode.APPEND, true, true)
      val margin = 15 // margin in mm
      val pageRectangle = PDRectangle.A4.margin(margin)

      val logoImageBytes = loadLogo("classpath:/digital-ai-release-logo.png")
      val logo = PDImageXObject.createFromByteArray(doc, logoImageBytes, "logo")
      val logoHeight = 10 * POINTS_PER_MM // 1cm
      val logoWidth = (logo.getWidth / logo.getHeight) * logoHeight
      val scaledLogoRectangle = new PDRectangle(0, 0, logoWidth, logoHeight)
      // move logo rectangle to the top left corner of the page
      val logoRectangle = pageRectangle.topLeft(scaledLogoRectangle)
      contentStream.drawImage(logo, logoRectangle)

      val contentParts = pngContent.split(",")
      val imageData = contentParts(1)
      val dashboardImageBytes = Base64.getDecoder.decode(imageData)
      val dashboard = PDImageXObject.createFromByteArray(doc, dashboardImageBytes, "dashboard")
      // new rectangle for free space after logo
      val freeSpaceRectangle = pageRectangle.verticallyAfter(logoRectangle)
      // create rectangle for scaled dashboard
      val scaledDashboardRectangle = freeSpaceRectangle.fitWidth(dashboard)
      // move rectangle for scaled dashboard to the top left corner of free space rectangle
      val dashboardRectangle = freeSpaceRectangle.topLeft(scaledDashboardRectangle)
      contentStream.drawImage(dashboard, dashboardRectangle)

      contentStream.close()

      val bufferSize = 1024 * 145 // ~ 144k is typical pdf size for this report
      doc.save(new BufferedOutputStream(outputStream, bufferSize))
    } finally {
      doc.close()
    }
  }

  private def loadLogo(location: String): Array[Byte] = {
    val loader = new DefaultResourceLoader(this.getClass.getClassLoader)
    val resource = loader.getResource(location)
    StreamUtils.copyToByteArray(resource.getInputStream)
  }

}

object PdfDashboardReportService {
  private val POINTS_PER_INCH = 72

  private val POINTS_PER_MM = 1 / (10 * 2.54f) * POINTS_PER_INCH

  implicit class ContentStreamExtensions(val contentStream: PDPageContentStream) extends AnyVal {
    // draws the image into the given rectangle
    def drawImage(image: PDImageXObject, rectangle: PDRectangle): Unit = {
      contentStream.drawImage(image, rectangle.getLowerLeftX, rectangle.getLowerLeftY, rectangle.getWidth, rectangle.getHeight)
    }
  }

  implicit class PDRectangleExtensions(val rectangle: PDRectangle) extends AnyVal {

    // create new rectangle with given margin in mm
    def margin(margin: Float): PDRectangle = {
      val result = new PDRectangle(rectangle.getLowerLeftX, rectangle.getLowerLeftY, rectangle.getWidth, rectangle.getHeight)
      result.setLowerLeftX(rectangle.getLowerLeftX + POINTS_PER_MM * margin)
      result.setLowerLeftY(rectangle.getLowerLeftY + POINTS_PER_MM * margin)
      result.setUpperRightX(rectangle.getUpperRightX - POINTS_PER_MM * margin)
      result.setUpperRightY(rectangle.getUpperRightY - POINTS_PER_MM * margin)
      result
    }

    // create new rectangle that occupies space required by inner rectangle from the top left corner of the enclosing rectangle
    def topLeft(innerRectangle: PDRectangle): PDRectangle = {
      new PDRectangle(rectangle.getLowerLeftX,
        rectangle.getLowerLeftY + rectangle.getHeight - innerRectangle.getHeight,
        innerRectangle.getWidth,
        innerRectangle.getHeight)
    }

    // create new rectangle that comes vertically after other rectangle
    def verticallyAfter(otherRectangle: PDRectangle): PDRectangle = {
      new PDRectangle(rectangle.getLowerLeftX, rectangle.getLowerLeftY, rectangle.getWidth, rectangle.getHeight - otherRectangle.getHeight)
    }

    // create new rectangle for image fitted by width
    def fitWidth(image: PDImageXObject): PDRectangle = {
      val scaled = Box(image.getWidth, image.getHeight).fitWidth(Box(rectangle.getWidth, rectangle.getHeight))
      new PDRectangle(rectangle.getLowerLeftX, rectangle.getLowerLeftY, scaled.width, scaled.height)
    }
  }

  case class Box(width: Float, height: Float) {

    def fitWidth(boundary: Box): Box = {
      val scale = width / height
      var scaled = Box(boundary.width, boundary.width / scale)
      if (scaled.height > boundary.height) {
        scaled = Box(scaled.height * this.width / this.height, boundary.height)
      }
      scaled
    }
  }

}
