package com.xebialabs.xlrelease.reports.job.handlers

import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener, Subscribe}
import com.xebialabs.xlrelease.reports.job.api.ReportingEngineService.ReportJobId
import com.xebialabs.xlrelease.reports.job.events._
import com.xebialabs.xlrelease.reports.job.events.internal.{FinishedRunReportJob, ReportJobInstanceEnqueued, ReportJobInstanceEvent}
import com.xebialabs.xlrelease.reports.job.impl.{AbortReportJob, ReportJobInstance}
import org.springframework.stereotype.Component

import java.util.concurrent.ConcurrentHashMap

@Component
@EventListener
class ReportJobInstanceEventHandler {

  private val locallyRunningReportJobs: ConcurrentHashMap[ReportJobId, ReportJobInstance] = new ConcurrentHashMap()

  @AsyncSubscribe
  def onAbortReportJob(event: AbortReportJob): Unit = {
    val localJobInstance = this.locallyRunningReportJobs.get(event.reportJobId)
    if (null != localJobInstance) {
      localJobInstance.abort()
    }
  }

  @Subscribe
  def onReportJobInstanceEvent(event: ReportJobInstanceEvent): Unit = {
    event match {
      case ReportJobInstanceEnqueued(reportJobInstance) =>
        registerReportJobInstance(reportJobInstance)
      case FinishedRunReportJob(reportJobId) =>
        unregisterReportJobInstance(reportJobId)
    }
  }

  @Subscribe
  def onReportJobDone(event: ReportJobEvent): Unit = {
    val jobDone = event match {
      case ReportJobCompletedEvent(_) => true
      case ReportJobFailedEvent(_) => true
      case ReportJobAbortedEvent(_) => true
      case ReportJobDeletedEvent(_) => true
      case ReportJobSubmittedEvent(_) => false
    }
    if (jobDone) {
      val jobId = event.reportJob.getJobId()
      unregisterReportJobInstance(jobId)
    }
  }

  private def registerReportJobInstance(reportJobInstance: ReportJobInstance): Unit = {
    locallyRunningReportJobs.put(reportJobInstance.getJobId, reportJobInstance)
  }

  private def unregisterReportJobInstance(jobId: ReportJobId): Unit = {
    locallyRunningReportJobs.remove(jobId)
  }
}
