package com.xebialabs.xlrelease.reports.job.impl.s3

import com.xebialabs.xlrelease.reports.job.api.ReportingEngineService.ReportJobId
import com.xebialabs.xlrelease.reports.job.api.{ReportResult, ReportStorage, StreamingReportResult}
import com.xebialabs.xlrelease.reports.job.domain.ReportJob
import com.xebialabs.xlrelease.storage.local.LocalStorageConfig
import com.xebialabs.xlrelease.storage.s3.{S3Storage, S3StorageConfig}
import com.xebialabs.xlrelease.storage.s3.S3Storage.{StringExtension, UriExtension}
import grizzled.slf4j.Logging

import java.io.File
import java.net.URI
import java.text.SimpleDateFormat
import java.util.Date

class S3ReportStorage(private val localStorageConfig: LocalStorageConfig,
                      private[s3] val s3StorageConfig: S3StorageConfig)
  extends ReportStorage with Logging {

  private val localWorkDir: File = localStorageConfig.basePath.toFile

  private[s3] val s3Storage = new S3Storage(s3StorageConfig)

  private implicit def implicitStorageConfig: S3StorageConfig = s3StorageConfig

  if (!localWorkDir.exists) {
    try {
      localWorkDir.mkdirs()
      logger.debug(s"Created root directory for reports at '${localWorkDir.getAbsolutePath}' based on provided location '${localStorageConfig.basePath}'")
    } catch {
      case e: Exception =>
        val msg = s"Unable to create root directory for reports '${localStorageConfig.basePath}'."
        throw new IllegalStateException(msg, e)
    }
  }

  if (!localWorkDir.isDirectory) {
    val msg = s"Report storage root directory location '${localStorageConfig.basePath}' does not point to a directory."
    throw new IllegalStateException(msg)
  }

  override def reportStorage(jobId: ReportJobId, generatedOn: Date): S3StreamingReportStorage = {
    val jobReportKey = getReportKey(jobId, generatedOn)
    val reportWorkdir = new File(localWorkDir, jobReportKey)
    if (!reportWorkdir.exists) {
      val isCreated = try {
        reportWorkdir.mkdirs()
      } catch {
        case e: Exception =>
          val msg = s"Unable to create directory for report job '$jobId' in '$localWorkDir'."
          throw new IllegalStateException(msg, e)
      }
      if (!isCreated) {
        throw new IllegalStateException(s"Unable to create directory for report job '$jobId' in '$localWorkDir'.")
      }
    }
    new S3StreamingReportStorage(this, reportWorkdir)
  }

  override def resolve(reportResult: ReportResult): StreamingReportResult = {
    reportResult match {
      case streamingReportResult: StreamingReportResult => streamingReportResult
      case _ =>
        val jobStorage = reportStorage(reportResult.jobId, reportResult.generatedOn)
        val jobDir = jobStorage.reportDir
        new S3StreamingReportResult(this, jobDir, reportResult)
    }
  }

  override def cleanup(reportJob: ReportJob): Boolean = {
    val jobReportKey = getReportKey(reportJob.getJobId(), reportJob.getEndTime)
    val result = s3Storage.cleanup(jobReportKey.toKeyUri)
    if (!result) {
      logger.warn(s"Failure while deleting report's job $jobReportKey directory")
    }
    result
  }

  private def getReportKey(jobId: ReportJobId, generatedOn: Date): String = {
    val formatter = new SimpleDateFormat("yyyy/MM/dd")
    val dateFragment = formatter.format(generatedOn)
    s"$dateFragment/$jobId".toKey
  }
}

object S3ReportStorage {

  private[s3] implicit class LocalFileExtension(localFile: File) {

    def toFileKey(implicit s3ReportStorage: S3ReportStorage): String = toReportUri(s3ReportStorage).toFileKey(s3ReportStorage.s3StorageConfig)

    def toKey(implicit s3ReportStorage: S3ReportStorage): String = toReportUri(s3ReportStorage).toKey(s3ReportStorage.s3StorageConfig)

    private def toReportUri(s3ReportStorage: S3ReportStorage): URI = s3ReportStorage.localStorageConfig.basePath.toUri.relativize(localFile.toURI)
  }
}
