package com.xebialabs.xlrelease.reports.job.repository.sql

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.reports.job.domain.ReportJob
import com.xebialabs.xlrelease.reports.job.repository.ReportJobFilters
import com.xebialabs.xlrelease.reports.job.repository.sql.ReportJobSchema.{REPORT_JOBS => RJ}
import com.xebialabs.xlrelease.repository.query._
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate
import org.springframework.util.StringUtils
import org.threeten.extra.Interval

import java.util.Date
import scala.jdk.CollectionConverters._

class SqlReportJobFilterQueryBuilder(val dialect: Dialect, val namedTemplate: NamedParameterJdbcTemplate, val sqlRepositoryAdapter: SqlRepositoryAdapter)
  extends FiltersQueryBuilder[ReportJobFilters, ReportJob] with FilterQueryBuilderSupport[ReportJobFilters, ReportJob] with ReportJobMapper {
  private lazy val queryTemplate =
    s"""
       |SELECT ${RJ.JOB_ID}, ${RJ.REPORT_TYPE},
       |       ${RJ.REPORT_NAME}, ${RJ.STATUS},
       |       ${RJ.NODE},
       |       ${RJ.TOTAL_WORK_ITEMS}, ${RJ.COMPLETED_WORK_ITEMS},
       |       ${RJ.SUBMIT_TIME}, ${RJ.START_TIME}, ${RJ.END_TIME},
       |       ${RJ.USERNAME}, ${RJ.RESULT_URI}, ${RJ.REPORT_DEFINITION}
       | FROM ${RJ.TABLE}
       | $whereClause
       | $orderClause
    """.stripMargin.linesIterator.filter(!_.trim.isEmpty).mkString(s"$NL")
  private lazy val totalQueryTemplate =
    s"""
       |SELECT COUNT(1)
       | FROM ${RJ.TABLE}
       | $whereClause
    """.stripMargin.linesIterator.filter(!_.trim.isEmpty).mkString(s"$NL")

  override def from(filter: ReportJobFilters): SelfType = {
    if (filter.jobIds != null && filter.jobIds.asScala.nonEmpty) {
      whereClauses += s"${RJ.JOB_ID} IN (:jobIds)"
      queryParams += "jobIds" -> filter.jobIds
    }
    if (!StringUtils.isEmpty(filter.reportType)) {
      whereClauses += s"${RJ.REPORT_TYPE} = :reportType"
      queryParams += "reportType" -> filter.reportType
    }
    if (!StringUtils.isEmpty(filter.reportName)) {
      like(RJ.REPORT_NAME, "reportName", filter.reportName)
    }
    if (filter.statuses != null && filter.statuses.asScala.nonEmpty) {
      whereClauses += s"${RJ.STATUS} IN (:statuses)"
      queryParams += "statuses" -> filter.statuses
    }
    if (!StringUtils.isEmpty(filter.username)) {
      like(RJ.USERNAME, "username", filter.username)
    }
    Option(filter.startTime).foreach { startTime =>
      searchInterval(RJ.START_TIME, startTime)
    }
    Option(filter.endTime).foreach { endTime =>
      searchInterval(RJ.END_TIME, endTime)
    }
    Option(filter.submitTime).foreach { submitTime =>
      searchInterval(RJ.SUBMIT_TIME, submitTime)
    }
    this
  }

  private def searchInterval(columnName: String, interval: Interval) = {
    if (interval.isUnboundedEnd) {
      whereClauses += s"$columnName >= :$columnName"
      queryParams += columnName -> Date.from(interval.getStart)
    } else if (interval.isUnboundedStart) {
      whereClauses += s"$columnName < :$columnName"
      queryParams += columnName -> Date.from(interval.getEnd)
    } else {
      whereClauses += s"$columnName >= :${columnName}_BEGIN AND $columnName < :${columnName}_END"
      queryParams += s"${columnName}_BEGIN" -> Date.from(interval.getStart)
      queryParams += s"${columnName}_END" -> Date.from(interval.getEnd)
    }
  }

  override def build(): PageableQuery[ReportJob] = {
    remapSortOrderParams()
    buildSortOrderClause("jobId")
    val resultsQueryString = pageableQuery(queryTemplate)
    val resultsQuery = new SqlListQuery[ReportJob](namedTemplate, resultsQueryString, queryParams.toMap, reportJobMapper)
    val totalCountQuery = new SqlQuery[Long](namedTemplate, totalQueryTemplate, queryParams.toMap, (rs, _) => rs.getLong(1))
    new SqlPageableQuery[ReportJob](namedTemplate, this.pageable, resultsQuery, totalCountQuery)
  }

  private def remapSortOrderParams(): Unit = {
    this.withSortParameters(
      "jobId" -> RJ.JOB_ID,
      "reportName" -> RJ.REPORT_NAME,
      "totalWorkItems" -> RJ.TOTAL_WORK_ITEMS,
      "completedWorkItems" -> RJ.COMPLETED_WORK_ITEMS,
      "submitTime" -> RJ.SUBMIT_TIME,
      "startTime" -> RJ.START_TIME,
      "endTime" -> RJ.END_TIME,
      "username" -> RJ.USERNAME
    )
  }
}
