package com.xebialabs.xlrelease.reports.job.impl.cleanup

import com.xebialabs.xlrelease.actors.cluster.XlrCluster.ClusterAddressOps
import com.xebialabs.xlrelease.reports.job.impl.ReportExecutorService
import com.xebialabs.xlrelease.reports.job.impl.cleanup.ReportCleanerActor.{CleanupDone, MissingNodesClusterCleanup, NodeCleanup}
import com.xebialabs.xlrelease.support.pekko.spring.SpringActor
import org.apache.pekko.actor.{Actor, ActorLogging, Stash}
import org.apache.pekko.cluster.{Cluster, Member}

import scala.concurrent.{ExecutionContextExecutorService, Future}
import scala.util.{Failure, Success}

object ReportCleanerActor {

  sealed trait ReportCleanerCommand

  case object CleanupDone extends ReportCleanerCommand

  case class NodeCleanup(node: Member) extends ReportCleanerCommand

  case object MissingNodesClusterCleanup extends ReportCleanerCommand

  final val name: String = "reportCleaner"
}

@SpringActor
class ReportCleanerActor(reportCleanerOperations: ReportCleanerOperations, reportExecutor: ReportExecutorService)
  extends Actor with ActorLogging with Stash {

  private val cluster = Cluster(context.system)

  override def receive: Receive = waiting

  // TODO cleanup should be executed on it's own thread pool
  implicit val ec: ExecutionContextExecutorService = reportExecutor.executionContext

  private def waiting: Receive = {
    case NodeCleanup(node) =>
      context.become(working)
      cleanupNode(node)
    case MissingNodesClusterCleanup =>
      context.become(working)
      cleanupMissingNodes()
    case msg => log.info(s"Unexpected message received $msg")
  }

  private def working: Receive = {
    case CleanupDone =>
      unstashAll()
      context.become(waiting, discardOld = false)
    case NodeCleanup(_) =>
      log.debug(s"Node cleanup received while cleaning reports is in progress")
      stash()
    case MissingNodesClusterCleanup =>
      log.debug(s"Missing nodes cleanup received while cleaning reports is in progress")
      stash()
  }

  private def cleanupNode(node: Member): Unit = {
    val host = node.address.host.get
    val port = node.address.port.get
    val nodeName = s"$host:$port"
    Future {
      log.info(s"Cleanup of reports after node '$nodeName' was removed from cluster started")
      reportCleanerOperations.nodeCleanup(nodeName)
      log.info(s"Cleanup of reports after node '$nodeName' was removed from cluster ended")
    } onComplete {
      case Success(_) => self ! CleanupDone
      case Failure(ex) =>
        log.error(s"Exception occurred while cleaning reports after node '$nodeName' was removed from cluster", ex)
        self ! CleanupDone
    }
  }

  private def cleanupMissingNodes(): Unit = {
    log.info(s"Cleanup of reports after passive cluster started")
    Future {
      log.info(s"Cleanup of reports after passive cluster started")
      val currentNodeIds = cluster.state.members.map(_.address).map(_.nodeAddress).toSeq
      reportCleanerOperations.cleanupMissingNodes(currentNodeIds)
      log.info(s"Cleanup of reports after passive cluster ended")
    } onComplete {
      case Success(_) => self ! CleanupDone
      case Failure(ex) =>
        log.error(s"Exception occurred while cleaning reports after passive cluster", ex)
        self ! CleanupDone
    }
  }

}


