package com.xebialabs.xlrelease.scm.connector

import com.xebialabs.xlrelease.domain.UserProfile
import com.xebialabs.xlrelease.domain.scm.connector.BitBucketCloudScmConnectorConfig
import com.xebialabs.xlrelease.scm.connector.HttpClientRequest.{MULTIPART_BODY_FORM, _}
import com.xebialabs.xlrelease.scm.data.{GitTag, ValidatedCommitInfo}
import org.apache.http.HttpHeaders
import org.springframework.http.{HttpStatus, ResponseEntity}

import java.nio.file.Paths
import scala.beans.BeanProperty
import scala.util.Try

class BitBucketCloudScmConnector(config: BitBucketCloudScmConnectorConfig) extends RestApiScmConnector(config) {

  override protected def testRepository: Try[Unit] = {
    bitBucketRequest
      .get(s"/repositories/${config.repository}/refs")
      .doRequest[Void]()
      .flatMap(_.tried)
      .map(_ => ())
  }

  override protected def commitAndTag(blobs: ScmBlobs, commitInfo: ValidatedCommitInfo, user: UserProfile): Try[String] = {
    for {
      gitCommitResponse <- bitbucketCommit(blobs, commitInfo.message)
      commitSha <- getCommitSha(gitCommitResponse)
      _ <- createTag(commitInfo.tag, commitSha)
    } yield commitSha
  }

  override protected def tagNotPresent(tag: String): Try[Boolean] = {
    bitBucketRequest
      .get(s"/repositories/${config.repository}/refs/tags/$tag")
      .doRequest[Void]().map(_.getStatusCode == HttpStatus.NOT_FOUND)
  }

  private def bitBucketRequest: ScmHttpClientWrapper = {
    HttpClientRequest
      .newRequest(config.restApiUrl)
      .setUriEncoded(false)
      .withAuth(config.credential)
  }

  private def bitbucketCommit(blobs: ScmBlobs, message: String): Try[ResponseEntity[Void]] = {
    val requestBuilder = bitBucketRequest
      .setType(MULTIPART_BODY_FORM())
      .addFormData("branch", config.branch)
      .addFormData("message", message)

    for (file <- blobs.filesToAdd) {
      requestBuilder.addFile(
        file.absolutePath,
        Paths.get(file.absolutePath).getFileName.toString,
        file.getContent
      )
    }

    requestBuilder.post(s"/repositories/${config.repository}/src")
      .doRequest[Void]().flatMap(_.checkResponse("Error occurred while trying to commit on BitBucket"))
  }

  private def getCommitSha(response: ResponseEntity[Void]): Try[String] = Try(response.getHeaders.toSingleValueMap.get(HttpHeaders.LOCATION).split("/").last)

  private def createTag(tag: GitTag, commitSha: String): Try[ResponseEntity[Void]] = {
    val requestBody = new BitBucketTag(tag.refName)
    requestBody.target.hash = commitSha

    bitBucketRequest
      .post(s"/repositories/${config.repository}/refs/tags", requestBody)
      .doRequest[Void]().flatMap(_.checkResponse(s"Error occurred while trying to create tag"))
  }
}

class BitBucketTag(@BeanProperty var name: String) {
  @BeanProperty
  var target: BitBucketTagHash = new BitBucketTagHash()
}

class BitBucketTagHash {
  @BeanProperty
  var hash: String = _
}

class BitBucketResponse[T] {
  @BeanProperty
  var values: Array[T] = _
}

class BitBucketFileInfo {
  @BeanProperty
  var path: String = _
}
