package com.xebialabs.xlrelease.scm.connector

import com.google.common.net.UrlEscapers
import com.xebialabs.xlrelease.domain.UserProfile
import com.xebialabs.xlrelease.domain.scm.connector.GitLabScmConnectorConfig
import com.xebialabs.xlrelease.domain.utils.ScmException
import com.xebialabs.xlrelease.scm.connector.GitLabCommitAction.{BASE64_ENCODING, CREATE_ACTION, UPDATE_ACTION}
import com.xebialabs.xlrelease.scm.connector.HttpClientRequest._
import com.xebialabs.xlrelease.scm.data.ValidatedCommitInfo
import org.springframework.http.{HttpStatus, ResponseEntity}

import java.util.{Base64, Collection => JCollection}
import scala.beans.BeanProperty
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._
import scala.util.Try

class GitLabScmConnector(config: GitLabScmConnectorConfig) extends RestApiScmConnector(config) {
  override protected def testRepository: Try[Unit] = {
    gitLabRequest
      .get(s"/projects/$repoName")
      .doRequest[Void]()
      .flatMap(_.tried)
      .map(_ => ())
  }

  override protected def commitAndTag(blobs: ScmBlobs, commitInfo: ValidatedCommitInfo, user: UserProfile): Try[String] = {
    for {
      _ <- createBranch()
      fileCommitAction <- createFileRequestAction(blobs)
      requestBody <- Try(new GitLabCommit(config.branch, commitInfo.message, fileCommitAction.asJavaCollection))
      gitCommitInfo <- createCommit(requestBody)
      _ <- createTag(commitInfo.tag.refName, gitCommitInfo.id)
    } yield gitCommitInfo.id
  }

  override protected def tagNotPresent(tag: String): Try[Boolean] = {
    gitLabRequest
      .get(s"/projects/$repoName/repository/tags/$tag")
      .doRequest[Void]()
      .map(_.getStatusCode == HttpStatus.NOT_FOUND)
  }

  private def createBranch(): Try[Unit] = Try {
    val projectResponse = gitLabRequest
      .get(s"/projects/$repoName")
      .doRequest[GitLabProjectResponse]()
      .get

    if (!projectResponse.isSuccessful) {
      throw ScmException("Unable to retrieve default branch", null, projectResponse.getStatusCode.value())
    }

    val exist = gitLabRequest
      .get(s"/projects/$repoName/repository/branches/${config.branch}")
      .doRequest[Void]()
      .get
      .isSuccessful

    if (!exist) {
      val response = gitLabRequest
        .addQueryParam("ref", projectResponse.getBody.default_branch)
        .addQueryParam("branch", config.branch)
        .post(s"/projects/$repoName/repository/branches")
        .doRequest[Void]()
        .get

      if (!response.isSuccessful) {
        throw ScmException("Unable to create branch", null, response.getStatusCode.value())
      }
    }
  }

  private def createCommit(requestBody: GitLabCommit): Try[GitLabCommitResponse] =
    gitLabRequest
      .post(s"/projects/$repoName/repository/commits", requestBody)
      .doRequest[GitLabCommitResponse]()
      .flatMap(_.checkResponse("Error occured while trying to commit on GitLab"))
      .map(_.getBody)

  private def createTag(tagName: String, sha: String): Try[ResponseEntity[Void]] = {
    gitLabRequest
      .addQueryParam("tag_name", tagName)
      .addQueryParam("ref", sha)
      .post(s"/projects/$repoName/repository/tags")
      .doRequest[Void]().flatMap(_.checkResponse("Error occurred while trying to create tag"))
  }

  private def createFileRequestAction(blobs: ScmBlobs): Try[Seq[GitLabCommitAction]] = Try {
    val actions = ListBuffer.empty[GitLabCommitAction]

    blobs.filesToAdd.foreach { file =>
      val response = gitLabRequest
        .addQueryParam("ref", config.branch)
        .get(s"/projects/$repoName/repository/files/${urlEncode(file.absolutePath)}")
        .doRequest[Void]()
        .get

      if (response.getStatusCode == HttpStatus.UNAUTHORIZED) {
        throw ScmException("Unable to connect to the repository.", null, response.getStatusCode.value())
      }

      if (response.isSuccessful) {
        actions += new GitLabCommitAction(
          UPDATE_ACTION,
          file.absolutePath,
          Base64.getEncoder.encodeToString(file.getContent()),
          BASE64_ENCODING
        )
      } else {
        actions += new GitLabCommitAction(
          CREATE_ACTION,
          file.absolutePath,
          Base64.getEncoder.encodeToString(file.getContent()),
          BASE64_ENCODING
        )
      }
    }

    actions.toSeq
  }

  private def repoName: String = urlEncode(config.repository)

  private def urlEncode(value: String): String = UrlEscapers.urlPathSegmentEscaper.escape(value)

  private def gitLabRequest = {
    HttpClientRequest
      .newRequest(config.restApiUrl)
      .setUriEncoded(true)
      .withAuth(config.credential)
  }
}

class GitLabTree {
  @BeanProperty
  var id: String = _
  @BeanProperty
  var name: String = _
}

object GitLabCommitAction {
  final val CREATE_ACTION = "create"
  final val UPDATE_ACTION = "update"
  final val BASE64_ENCODING = "base64"
}

class GitLabCommitAction(@BeanProperty var action: String,
                         @BeanProperty var file_path: String,
                         @BeanProperty var content: String,
                         @BeanProperty var encoding: String = "text")

class GitLabCommit(@BeanProperty var branch: String,
                   @BeanProperty var commit_message: String,
                   @BeanProperty var actions: JCollection[GitLabCommitAction])

class GitLabCommitResponse {
  @BeanProperty
  var id: String = _
}

class GitLabProjectResponse {
  @BeanProperty
  var default_branch: String = _
}

