package com.xebialabs.xlrelease.scm.connector

import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.UserProfile
import com.xebialabs.xlrelease.domain.scm.connector.JGitConnectorConfig
import com.xebialabs.xlrelease.domain.utils.ScmException
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.scm.connector.JGitConnector.{deleteRepo, getFullBranchName, getLocalDirPath}
import com.xebialabs.xlrelease.scm.data.ValidatedCommitInfo
import com.xebialabs.xlrelease.support.config.TypesafeConfigExt._
import grizzled.slf4j.Logging
import org.eclipse.jgit.api.ResetCommand.ResetType
import org.eclipse.jgit.api.errors.{InvalidRemoteException, JGitInternalException, TransportException}
import org.eclipse.jgit.api.{Git, PullResult}
import org.eclipse.jgit.errors
import org.eclipse.jgit.errors.NoRemoteRepositoryException
import org.eclipse.jgit.internal.JGitText
import org.eclipse.jgit.lib._
import org.eclipse.jgit.merge.MergeStrategy
import org.eclipse.jgit.transport.RemoteRefUpdate.Status._
import org.eclipse.jgit.transport.{RefSpec, RemoteRefUpdate}
import org.eclipse.jgit.util.{FS, FileUtils}

import java.io.File
import java.nio.file.{Files, Paths}
import java.util
import java.util.Objects
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

object JGitConnector extends Logging {
  lazy val DEFAULT_SCM_WORKDIR: String = XlrConfig.getInstance.repository.workDir + "/scm_repos"

  def verifyRepositoryPath(url: String): Unit = {
    // URL is of the format http(s)://url/project/repo
    if (url == null || getOrgRepoName(url).length != 2) {
      throw ScmException(s"Invalid repository format $url")
    }
  }

  def deleteRepo(configId: String): Unit = {
    val dir = new File(getWorkdirPath)
    if (dir.exists()) {
      dir.listFiles().filter(_.isDirectory).filter(_.getName.startsWith(Ids.getName(configId))).foreach(f => {
        Try {
          FileUtils.delete(
            f,
            FileUtils.RECURSIVE | FileUtils.RETRY | FileUtils.SKIP_MISSING
          )
        }.recover {
          case ex => logger.error(s"Unable to delete repo $getWorkdirPath/${f.getName}", ex)
        }
      })
    }
  }

  def getConfigDirPath(configId: String): String = {
    getWorkdirPath + s"/${Ids.getName(configId)}"
  }

  def getFullBranchName(branch: String): String = {
    Constants.R_HEADS + branch
  }

  def getLocalDirPath(configId: String, repoUrl: String, branch: String): String = {
    var repoName = getOrgRepoName(repoUrl).mkString("_")
    // added to handle : in the directory path of windows
    repoName = repoName.replace(":", "_")
    s"$getWorkdirPath/${Ids.getName(configId)}_${repoName}_$branch"
  }

  private def getOrgRepoName(url: String): Array[String] = url.split("/").takeRight(2)

  private def getWorkdirPath: String = {
    XlrConfig.getInstance.getXl().getOptionalConfig("features")
      .map(_.getString("scm.workdir-path", DEFAULT_SCM_WORKDIR))
      .getOrElse(DEFAULT_SCM_WORKDIR)
  }

  def isFileRepo(url: String): Boolean = url.startsWith("file")

}

trait JGitConnector extends ScmConnector with Logging {
  def getConfigId: String

  def getConnectionSettings: GitConnectionSettings

  lazy val fullBranchName: String = getFullBranchName(getConnectionSettings.getBranch)
  // The repodir is unique for (scm configuration, repository, branch)
  lazy val repoDirPath: String = getLocalDirPath(
    getConfigId,
    getConnectionSettings.getUrl,
    getConnectionSettings.getBranch
  )

  private var _git: Git = _

  private var branches: mutable.Map[String, Ref] = _
  private var tags: mutable.Map[String, Ref] = _

  override def close(): Unit = {
    if (_git != null) {
      _git.close()
      _git = null
    }
    tags = null
    branches = null
    GitClient.removeCurrentGitConnectionSettings()
  }

  override def getIdentifier(uniqueString: String): String = {
    s"${getConnectionSettings.getUrl}/$uniqueString"
  }

  // fixme: this doesn't check if config is authenticated to push
  override def connectionStatus(): Try[Unit] = {
    val refs = for {
      _ <- verifyRepositoryPath()
      remoteRefs <- Try(lsRemoteRepository(getConnectionSettings.getUrl, heads = true, tags = true))
    } yield remoteRefs
    refs.flatMap { r =>
      val (b, t) = r.partition(_._1.startsWith(Constants.R_HEADS))
      branches = b
      tags = t
      if (branches.isEmpty) {
        Failure(ScmException(s"The remote repository is empty."))
      } else if (!branches.contains(fullBranchName)) {
        Failure(ScmException(s"Branch ${getConnectionSettings.getBranch} does not exist in the remote repository."))
      } else {
        Success(())
      }
    }
  }

  override protected def tagNotPresent(tag: String): Try[Boolean] = Try {
    if (tags == null) {
      tags = lsRemoteRepository(getConnectionSettings.getUrl, heads = false, tags = true)
    }
    !tags.contains(Constants.R_TAGS + tag)
  }

  def getAllBranches: Seq[String] = {
    if (branches == null) {
      branches = lsRemoteRepository(getConnectionSettings.getUrl, heads = true, tags = false)
    }
    branches.keys.toSeq.map(_.stripPrefix(Constants.R_HEADS))
  }

  //noinspection ScalaStyle
  override protected def commitAndTag(blobs: ScmBlobs,
                                      commitInfo: ValidatedCommitInfo,
                                      user: UserProfile): Try[String] = Try {
    val git = if (getRepoMetadataDirRef.isDefined) {
      val tmpGit = getOrInitGit()
      gitHardResetToOrigin(tmpGit)
      // get latest commit
      pullWithRecovery(tmpGit)
      tmpGit
    } else {
      // avoid additional pull in case repo doesn't exist locally
      getOrInitGit()
    }

    val repo = git.getRepository

    val repoParentDir = repo.getDirectory.getParent
    // ensure directories exist in repository
    // todo optimize: don't mkdirs for every filepath
    blobs.filesToAdd.foreach { file =>
      val path = Paths.get(repoParentDir, file.absolutePath)
      val folder = new File(path.getParent.toString)

      folder.mkdirs()
      // write
      Files.write(path, file.getContent())
    }

    // add the files
    git.add().addFilepattern(".").call()
    // delete files
    if (blobs.fileToRemove.nonEmpty) {
      val removeCommand = git.rm()
      blobs.fileToRemove.foreach { fileName =>
        val path = Paths.get(repoParentDir, fileName)
        Files.deleteIfExists(path)
        removeCommand.addFilepattern(fileName)
      }
      removeCommand.call()
    }
    // commit
    // committer - the user that is authenticated to remote repository
    // author - the XLR user
    val revCommit = git.commit()
      .setCommitter(getCommitter(user))
      .setAuthor(getAuthor(user))
      .setMessage(commitInfo.message)
      .call()
    // tag
    val tagRef = git.tag()
      .setAnnotated(false)
      .setObjectId(revCommit)
      .setName(commitInfo.tag.refName)
      .call()

    // push
    val pushStatusReference = pushWithRecovery(git, fullBranchName, tagRef.getName)
    val pushStatus = pushStatusReference.getStatus
    // handle failed push
    pushStatus match {
      case OK => ()
      case REJECTED_NONFASTFORWARD | REJECTED_REMOTE_CHANGED =>
        // Re-attempt pushing once
        pullWithRecovery(git, tagRef.getName)
        val reAttemptPushStatusRef = pushWithRecovery(git, fullBranchName, tagRef.getName)
        val reAttemptPushStatus = reAttemptPushStatusRef.getStatus
        if (reAttemptPushStatus == REJECTED_NONFASTFORWARD
          || reAttemptPushStatus == REJECTED_REMOTE_CHANGED
          || reAttemptPushStatus == NON_EXISTING
          || reAttemptPushStatus == REJECTED_OTHER_REASON) {
          gitHardResetToOrigin(git, tagRef.getName)
          throw ScmException(
            s"""Unable to push to remote repository.
               |  Status: [${reAttemptPushStatus.name()}]
               |  Exception: [${reAttemptPushStatusRef.getMessage}]""".stripMargin)
        }
      case _ =>
        gitHardResetToOrigin(git, tagRef.getName)
        throw ScmException(
          s"""Unable to push to remote repository.
             |  Status: [${pushStatus.name()}]
             |  Exception: [${pushStatusReference.getMessage}]""".stripMargin)
    }

    revCommit.getName
  }

  // isn't static because requires threadlocal to be set by connector init
  def verifyRemoteBranch(url: String, branchName: String, gitConfigTitle: Option[String] = None): Unit = {
    val remoteBranches = lsRemoteRepository(url, heads = true, tags = false, gitConfigTitle)
    if (!remoteBranches.contains(getFullBranchName(branchName))) {
      throw ScmException(s"Branch $branchName does not exist in the remote repository.")
    }
  }

  def getOrInitGit(gitConfigTitle: Option[String] = None): Git = {
    if (_git != null) {
      _git
    } else {
      try {
        _git = getRepoMetadataDirRef.fold(initRepo()) { dir =>
          val tmpGit = Git.open(dir)
          tmpGit
        }
        _git
      } catch {
        case _: InvalidRemoteException if gitConfigTitle.nonEmpty =>
          throw ScmException(repoNotFoundErrorMessage(getConnectionSettings.getUrl, gitConfigTitle))
        case t: Throwable =>
          throw ScmException("Error while establishing connection to the remote repository.", cause = unwrap(t))
      }
    }
  }

  protected def getRepoMetadataDirRef: Option[File] = {
    val repoGitMetadataDir = new File(s"$repoDirPath/.git")
    val gitRepoExists = RepositoryCache.FileKey.isGitRepository(repoGitMetadataDir, FS.DETECTED)
    if (gitRepoExists) {
      Some(repoGitMetadataDir)
    } else {
      None
    }
  }

  protected def gitHardResetToOrigin(git: Git, localTag: String = null): Unit = {
    try {
      git.reset()
        .setMode(ResetType.HARD)
        .setRef(s"${Constants.DEFAULT_REMOTE_NAME}/${getConnectionSettings.getBranch}")
        .call()
      if (localTag != null) {
        git.tagDelete().setTags(localTag).call()
      }
    } catch {
      case resetError: Throwable =>
        deleteRepo(getConfigId)
        throw resetError
    }
  }

  // isn't static because requires threadlocal to be set by connector init
  protected def lsRemoteRepository(url: String, heads: Boolean, tags: Boolean, gitConfigTitle: Option[String] = None): mutable.Map[String, Ref] = {
    try {
      Git.lsRemoteRepository()
        .setRemote(url)
        .setHeads(heads)
        .setTags(tags)
        .callAsMap()
        .asScala
    } catch {
      case t: Throwable =>
        t.getCause match {
          case _: NoRemoteRepositoryException => throw ScmException(repoNotFoundErrorMessage(getConnectionSettings.getUrl, gitConfigTitle))
          case _ => throw ScmException("Error retrieving remote repository details.", cause = unwrap(t))
        }
    }
  }

  protected def gitPullTry(git: Git): Try[PullResult] = {
    try {
      val result = git.pull()
        .setStrategy(MergeStrategy.THEIRS)
        .call()
      Success(result)
    } catch {
      case t: Throwable =>
        Failure(ScmException("Error while fetching changes.", cause = unwrap(t)))
    }
  }

  protected def gitPushTry(git: Git, branch: String, tag: String, isAtomic: Boolean = true): Try[RemoteRefUpdate] = {
    try {
      val refSpecs = java.util.List.of(new RefSpec(branch), new RefSpec(tag))
      val pushResultCommit = git.push()
        .setAtomic(isAtomic)
        .setRefSpecs(refSpecs)
        .call().asScala.toList.head
      Success(pushResultCommit.getRemoteUpdate(fullBranchName))
    } catch {
      case ex: TransportException if ex.getMessage.contains(JGitText.get.atomicPushNotSupported) =>
        logger.info(s"Remote server ${getConnectionSettings.getUrl} doesn't support atomic push, retrying with non-atomic push...")
        gitPushTry(git, branch, tag, isAtomic = false)
      case t: Throwable =>
        Failure(ScmException(s"Error while saving version [$tag].", cause = unwrap(t)))
    }
  }

  protected def pullWithRecovery(git: Git, localTag: String = null, gitConfigTitle: Option[String] = None): PullResult = {
    gitPullTry(git).recover(gitErrorRecovery(git, localTag, gitConfigTitle)).get
  }

  protected def pushWithRecovery(git: Git, branch: String, tag: String, gitConfigTitle: Option[String] = None): RemoteRefUpdate = {
    gitPushTry(git, branch, tag).recover(gitErrorRecovery(git, tag, gitConfigTitle)).get
  }

  protected def gitErrorRecovery[U](git: Git, localTag: String = null, gitConfigTitle: Option[String] = None): PartialFunction[Throwable, U] = {
    case t: Throwable =>
      t.getCause match {
        case _: InvalidRemoteException => {
          deleteRepo(getConfigId)
          throw ScmException(repoNotFoundErrorMessage(getConnectionSettings.getUrl, gitConfigTitle))
        }
        case _: JGitInternalException => deleteRepo(getConfigId)
        case _ =>
          gitHardResetToOrigin(git, localTag)
      }
      throw t
  }

  protected def initRepo(): Git = {
    // prepare local repository
    deleteRepo(getConfigId)

    val workDir = new File(repoDirPath)
    workDir.mkdirs()

    // the repository MUST exist on remote server
    Git.cloneRepository()
      .setURI(getConnectionSettings.getUrl)
      .setDirectory(workDir)
      .setBranchesToClone(util.Arrays.asList(fullBranchName))
      .setBranch(fullBranchName)
      .call()
  }

  protected def verifyRepositoryPath(): Try[Unit] = {
    Try(JGitConnector.verifyRepositoryPath(getConnectionSettings.getUrl))
  }

  protected def getCommitter(userProfile: UserProfile): PersonIdent = {
    val committerUsername = getConnectionSettings.getCommitterUsername
    val committerEmail = getConnectionSettings.getCommitterEmail

    if (isNotBlank(committerUsername) && isNotBlank(committerEmail)) {
      new PersonIdent(committerUsername, committerEmail)
    } else {
      getAuthor(userProfile)
    }
  }

  protected def getAuthor(userProfile: UserProfile): PersonIdent = {
    new PersonIdent(
      if (isNotBlank(userProfile.getFullName)) {
        userProfile.getFullName
      } else {
        userProfile.getCanonicalId
      },
      Objects.toString(userProfile.getEmail, "")
    )
  }

  protected def unwrap(t: Throwable): Throwable = {
    // Unwrap the internal git API exception if necessary
    val unwrappedError = t match {
      case ex: TransportException if ex.getCause.isInstanceOf[errors.TransportException] && t.getMessage == ex.getMessage =>
        ex.getCause
      case _ => t
    }

    unwrappedError match {
      case ex: errors.TransportException if ex.getMessage != null =>
        handleTransportException(ex)
      case _ => unwrappedError
    }
  }

  private def handleTransportException(ex: errors.TransportException): Exception = {
    val message = ex.getMessage

    if (message.contains(JGitText.get.noCredentialsProvider)) {
      ScmException("Invalid credentials provided. Check credentials and try again.", cause = ex)
    } else if (message.contains("git-receive-pack") || message.contains("git-upload-pack")) {
      ScmException("Insufficient permissions. Check permissions and try again.", cause = ex)
    } else {
      ex
    }
  }

  private def repoNotFoundErrorMessage(url: String, gitConfigTitle: Option[String]): String =
    s"The configured Git repository cannot be accessed. " +
      s"Check if the repository '$url' exists in Git, and update this folder’s versioning configuration settings [${gitConfigTitle.getOrElse("")}]. "

}

class DefaultJGitConnector(config: JGitConnectorConfig) extends JGitConnector {
  override val getConfigId: String = config.getId

  override val getConnectionSettings: GitConnectionSettings = new GitConnectionSettings(
    config.url,
    config.branch,
    config.authenticationMethod.name(),
    config.username,
    PasswordEncrypter.getInstance().ensureDecrypted(config.password),
    config.domain,
    config.proxyHost,
    config.proxyPort,
    config.proxyUsername,
    PasswordEncrypter.getInstance().ensureDecrypted(config.proxyPassword),
    config.proxyDomain,
    config.committerUsername,
    config.committerEmail
  )

  // For thread-local settings for Http (authentication settings)
  private var gitClient: GitClient = _
  try {
    gitClient = new GitClient(getConnectionSettings)
  } catch {
    case t: Throwable =>
      throw ScmException("Unable to initialize git client with current settings.", unwrap(t))
  }
}
