package com.xebialabs.xlrelease.status.service

import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.api.internal.views.{ExternalDeploymentOrderDirection, ExternalDeploymentOrderMode}
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence
import com.xebialabs.xlrelease.status.repository.persistence.ExternalDeploymentPersistence
import com.xebialabs.xlrelease.status.service.script.ExternalDeploymentScriptService
import com.xebialabs.xlrelease.status.webhook.configuration.StatusHttpConnection
import com.xebialabs.xlrelease.status.webhook.events.{StatusWebhookEventSource, UpdateStatusEvent}
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import java.lang
import java.util.{List => JList}
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

trait EndpointExternalDeploymentService {
  def getExternalDeployment(endpointIds: Vector[String],
                            maxAge: Long = 0,
                            page: lang.Long,
                            resultsPerPage: lang.Long,
                            orderBy: ExternalDeploymentOrderMode,
                            direction: ExternalDeploymentOrderDirection): EndpointExternalDeployment
}

@Service
class EndpointExternalDeploymentServiceImpl(
                                             configurationRepository: ConfigurationRepository,
                                             configurationPersistence: ConfigurationPersistence,
                                             externalDeploymentPersistence: ExternalDeploymentPersistence,
                                             statusScriptService: ExternalDeploymentScriptService
                                  ) extends EndpointExternalDeploymentService with Logging {

  override def getExternalDeployment(
                                      endpointIds: Vector[String],
                                      maxAge: Long,
                                      page: lang.Long,
                                      resultsPerPage: lang.Long,
                                      orderBy: ExternalDeploymentOrderMode,
                                      direction: ExternalDeploymentOrderDirection): EndpointExternalDeployment = {
    val endpointToConnectionMap = endpointIds.map(endpointId => {
      val statusWebhookEventSource = configurationRepository.read[StatusWebhookEventSource](endpointId)
      val connection = configurationRepository.read[StatusHttpConnection](statusWebhookEventSource.sourceServer.getId)
      if (!externalDeploymentPersistence.exists(endpointId, maxAge)) {
        Try(statusScriptService.executeScript(statusWebhookEventSource)) match {
          case Success(result: JList[UpdateStatusEvent]) =>
            externalDeploymentPersistence.delete(endpointId)
            externalDeploymentPersistence.save(endpointId, result.asScala.toVector)
          case Success(errorObject) =>
            error(s"Illegal response in ${connection.getId} statusScript response. Returned type was ${errorObject.getClass.getName}, expected list of states")
            throw EndpointExternalDeploymentError(connection)
          case Failure(exception) =>
            warn(s"Error fetching Application status on ${connection.getId} [${connection.getTitle}]:", exception)
            throw EndpointExternalDeploymentError(connection)
        }
      }
      configurationPersistence.getUid(endpointId).get ->
        ConnectionServerData(connection.getId, connection.getTitle, connection.getType.toString, connection.getUrl)
    }).toMap

    EndpointExternalDeployment(
      endpointToConnectionMap,
      externalDeploymentPersistence.findByEndpointId(endpointIds, resultsPerPage, resultsPerPage * page, orderBy, direction)
    )
  }
}

case class EndpointUpdateStatusEvent(endpointId: String, event: UpdateStatusEvent)

case class EndpointExternalDeployment(connectionServers: Map[Integer, ConnectionServerData],
                                      states: Vector[EndpointUpdateStatusEvent])

case class ConnectionServerData(connectionId: String, title: String, serverType: String, serverUrl: String)

case class EndpointExternalDeploymentError(ci: ConfigurationItem) extends DeployitException(ci)
