package com.xebialabs.xlrelease.status.service.script

import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.domain.Configuration
import com.xebialabs.xlrelease.script.XlrScriptContext
import com.xebialabs.xlrelease.script.builder.XlrScripContextExtensions
import com.xebialabs.xlrelease.script.jython.{JythonScriptService, XlrJythonSupport}
import com.xebialabs.xlrelease.service.AutoconfigExecutor
import com.xebialabs.xlrelease.status.service.script.StatusWebhookAutoconfigureExecutor.createAutoconfigScriptContext
import com.xebialabs.xlrelease.status.webhook.events.StatusWebhookEventSource
import com.xebialabs.xlrelease.views.AutoconfigResponse
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.util.{Failure, Success, Try}

@Component
class StatusWebhookAutoconfigureExecutor @Autowired()(
                                                       val scriptService: JythonScriptService
                                                     ) extends XlrJythonSupport with AutoconfigExecutor with Logging {
  override def supports(configuration: Configuration): Boolean =
    configuration.getType.instanceOf(Type.valueOf(classOf[StatusWebhookEventSource]))

  override def autoconfigure(configuration: Configuration): AutoconfigResponse = {
    configuration match {
      case webhook: StatusWebhookEventSource =>
        Try(executeScript(webhook)) match {
          case Success(result: AutoconfigResult) =>
            new AutoconfigResponse(true, result.message, result.details, result.url)
          case Success(errorObject) =>
            error(s"Illegal response in ${webhook.sourceServer.getId} autoconfiguration response." +
              s" Returned type was ${errorObject.getClass.getName}, expected autoconfig result")
            new AutoconfigResponse(false, "Could not resolve script result. Check logs for more details.", null, null)
          case Failure(exception) =>
            warn(s"Error executing autoconfiguration on ${webhook.sourceServer.getId} [${webhook.sourceServer.getTitle}]:", exception)
            new AutoconfigResponse(false, exception.getMessage, null, null)
        }
      case _ =>
        warn(s"Cannot execute autoconfiguration on type: ${configuration.getType.toString}:")
        throw AutoconfigureConfigurationError(configuration)
    }
  }

  private def executeScript(webhook: StatusWebhookEventSource): AnyRef = {
    logger.debug(s"Executing autoconfig script for StatusWebhookEventSource ${webhook.getName}")
    val scriptContext = createAutoconfigScriptContext(webhook)
    executeScript(scriptContext)
    scriptContext.getAttribute("data")
  }
}

private object StatusWebhookAutoconfigureExecutor {
  def createAutoconfigScriptContext(webhook: StatusWebhookEventSource): XlrScriptContext =
    new AutoconfigScriptContextBuilder(webhook).build()
}

private class AutoconfigScriptContextBuilder(
                                              webhook: StatusWebhookEventSource
                                        ) extends BaseScriptContextBuilder(webhook) {

  override val scriptProperty: String = AutoconfigScriptProperty

  override protected def doBuild(context: XlrScriptContext): Unit = {
    addConnectionServerAttributes(context)
    context.addCiAsMap("webhook", webhook)
    addScript(context)
  }
}

case class AutoconfigResult(message: String, details: String, url: String)

case class AutoconfigureConfigurationError(ci: ConfigurationItem) extends DeployitException(ci)
