package com.xebialabs.xlrelease.status.repository.persistence

import com.xebialabs.xlrelease.api.internal.views.{ExternalDeploymentOrderDirection, ExternalDeploymentOrderMode}
import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.CONFIGURATIONS
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, Utils}
import com.xebialabs.xlrelease.status.repository.persistence.ExternalDeploymentSchema.EXTERNAL_DEPLOYMENT
import com.xebialabs.xlrelease.status.service.EndpointUpdateStatusEvent
import com.xebialabs.xlrelease.status.webhook.events._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import java.util.Date
import scala.jdk.CollectionConverters._

@Repository
@IsTransactional
class ExternalDeploymentPersistenceImpl @Autowired()(override val configurationPersistence: ConfigurationPersistence)
                                                    (@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                                     @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect)
  extends SqlRepository with ExternalDeploymentPersistence with Logging {

  override def save(uid: ExternalDeploymentUid, deployment: ExternalDeployment): Unit =
    if (exists(uid)) {
      updateEndpointPath(uid, deployment)
    }
    else {
      insertEndpointPath(uid, deployment)
    }

  override def persist(uid: ExternalDeploymentUid, externalDeploymentEvent: ExternalDeploymentEvent): Unit =
    externalDeploymentEvent match {
      case _: DeleteStatusEvent
        if Option(uid.applicationCorrelationId).isDefined && Option(uid.destionationCorrelationId).isDefined =>
        delete(uid);
      case _: DeleteStatusEvent =>
        delete(uid.endpointId)
      case createEvent: CreateStatusEvent =>
        debug(s"Received CreateStatusEvent [${createEvent.applicationPath} on ${createEvent.destination}]- skipping save to database.")
      case updateStatus: UpdateStatusEvent =>
        save(uid, ExternalDeployment(updateStatus))
      case event => // all the ignored types for now
        warn(s"Couldn't execute ExternalDeploymentEvent[${event.getClass.getName}] persistence - skipping save to database.")
    }

  override def batchSaveOrUpdate(endpointId: CiId, states: Vector[UpdateStatusEvent]): Unit = {
    delete(endpointId)
    states.foreach(state =>
      save(ExternalDeploymentUid(endpointId, state.applicationUid, state.destinationUid), ExternalDeployment(state)))
  }
}

trait ExternalDeploymentPersistence  extends PersistenceSupport with Utils with LimitOffset  with Logging {
  override implicit val dialect: Dialect

  def persist(externalDeploymentUid: ExternalDeploymentUid, externalDeploymentEvent: ExternalDeploymentEvent): Unit

  def configurationPersistence: ConfigurationPersistence

  def save(externalDeploymentUid: ExternalDeploymentUid, deployment: ExternalDeployment): Unit

  def batchSaveOrUpdate(endpointId: CiId, states: Vector[UpdateStatusEvent]): Unit

  private final val STMT_APPLICATION_STATUS_EXISTS: String =
    s"""SELECT
       |  COUNT(1)
       | FROM ${CONFIGURATIONS.TABLE} conf
       | JOIN ${EXTERNAL_DEPLOYMENT.TABLE} ed ON conf.${CONFIGURATIONS.CI_UID} = ed.${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID}
       | WHERE conf.${CONFIGURATIONS.CI_UID} = :${CONFIGURATIONS.CI_UID} AND
       | ed.${EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID} = :${EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID} AND
       | ed.${EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID} = :${EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID}
       |""".stripMargin

  def exists(uid: ExternalDeploymentUid): Boolean =
    sqlQuery(
      STMT_APPLICATION_STATUS_EXISTS,
      params(
        CONFIGURATIONS.CI_UID -> configurationPersistence.getUid(uid.endpointId).get,
        EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID -> uid.applicationCorrelationId,
        EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID -> uid.destionationCorrelationId
      ),
      _.getInt(1) > 0
    ).head

  private final val STMT_APPLICATION_CHECK_AGE: String =
    s"""SELECT
       |  COUNT(1)
       | FROM ${CONFIGURATIONS.TABLE} conf
       | JOIN ${EXTERNAL_DEPLOYMENT.TABLE} e ON conf.${CONFIGURATIONS.CI_UID} = e.${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID}
       | WHERE conf.${CONFIGURATIONS.CI_UID} = :${CONFIGURATIONS.CI_UID}
       | AND e.${EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME} > :${EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME}
       |""".stripMargin

  def exists(endpointId: CiId, maxAge: Long): Boolean =
    sqlQuery(
      STMT_APPLICATION_CHECK_AGE,
      params(
        CONFIGURATIONS.CI_UID -> configurationPersistence.getUid(endpointId).get,
        EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME -> new Date(System.currentTimeMillis - maxAge),
      ),
      _.getInt(1) > 0
    ).head

  private final val STMT_COUNT_BY_ENDPOINT_IDS_AND_FILTER =
    s"""SELECT COUNT(1)
       | FROM ${CONFIGURATIONS.TABLE} conf
       | JOIN ${EXTERNAL_DEPLOYMENT.TABLE} e ON conf.${CONFIGURATIONS.CI_UID} = e.${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID}
       | WHERE e.${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID} in (:configurationUids)
       |  AND (
       |    LOWER(e.${EXTERNAL_DEPLOYMENT.APPLICATION_NAME}) LIKE LOWER(:condition)
       |    OR LOWER(e.${EXTERNAL_DEPLOYMENT.DESTINATION}) LIKE LOWER(:condition)
       |    OR LOWER(e.${EXTERNAL_DEPLOYMENT.NAMESPACE}) LIKE LOWER(:condition)
       |    OR LOWER(e.${EXTERNAL_DEPLOYMENT.STATUS}) LIKE LOWER(:condition)
       |    OR LOWER(e.${EXTERNAL_DEPLOYMENT.VERSION_TAG}) LIKE LOWER(:condition)
       | )
       |""".stripMargin

  def count(endpointIds: Vector[CiId], condition: String):  Int = {
    val endpointUids = endpointIds.map(endpointId => configurationPersistence.getUid(endpointId).get).toList.asJava
    sqlQuery(
      STMT_COUNT_BY_ENDPOINT_IDS_AND_FILTER,
      params("configurationUids" -> endpointUids, "condition" -> s"%${condition}%"),
      _.getInt(1)
    ).head
  }


  private final val STMT_BASE_FIND = s"""SELECT
       |  e.${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID} as ${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID},
       |  e.${EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID} as ${EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID},
       |  e.${EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID} as ${EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID},
       |  e.${EXTERNAL_DEPLOYMENT.DESTINATION} as ${EXTERNAL_DEPLOYMENT.DESTINATION},
       |  e.${EXTERNAL_DEPLOYMENT.APPLICATION_NAME} as ${EXTERNAL_DEPLOYMENT.APPLICATION_NAME},
       |  e.${EXTERNAL_DEPLOYMENT.APPLICATION_PATH} as ${EXTERNAL_DEPLOYMENT.APPLICATION_PATH},
       |  e.${EXTERNAL_DEPLOYMENT.NAMESPACE} as ${EXTERNAL_DEPLOYMENT.NAMESPACE},
       |  e.${EXTERNAL_DEPLOYMENT.STATUS_GROUP} as ${EXTERNAL_DEPLOYMENT.STATUS_GROUP},
       |  e.${EXTERNAL_DEPLOYMENT.STATUS} as ${EXTERNAL_DEPLOYMENT.STATUS},
       |  e.${EXTERNAL_DEPLOYMENT.VERSION_TAG} as ${EXTERNAL_DEPLOYMENT.VERSION_TAG},
       |  e.${EXTERNAL_DEPLOYMENT.VERSION_STATE} as ${EXTERNAL_DEPLOYMENT.VERSION_STATE},
       |  e.${EXTERNAL_DEPLOYMENT.DEPLOYMENT_TYPE} as ${EXTERNAL_DEPLOYMENT.DEPLOYMENT_TYPE},
       |  e.${EXTERNAL_DEPLOYMENT.USER} as ${EXTERNAL_DEPLOYMENT.USER},
       |  e.${EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME} as ${EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME}
       | FROM ${CONFIGURATIONS.TABLE} conf
       | JOIN ${EXTERNAL_DEPLOYMENT.TABLE} e ON conf.${CONFIGURATIONS.CI_UID} = e.${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID}"""

  private final val STMT_FIND_BY_ENDPOINT_IDS =
    s"""${STMT_BASE_FIND}
       | WHERE e.${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID} in (:configurationUids)
       | AND (
       |    LOWER(e.${EXTERNAL_DEPLOYMENT.APPLICATION_NAME}) LIKE LOWER(:condition)
       |    OR LOWER(e.${EXTERNAL_DEPLOYMENT.DESTINATION}) LIKE LOWER(:condition)
       |    OR LOWER(e.${EXTERNAL_DEPLOYMENT.NAMESPACE}) LIKE LOWER(:condition)
       |    OR LOWER(e.${EXTERNAL_DEPLOYMENT.STATUS}) LIKE LOWER(:condition)
       |    OR LOWER(e.${EXTERNAL_DEPLOYMENT.VERSION_TAG}) LIKE LOWER(:condition)
       | )
       |""".stripMargin

  def findForEndpoints(endpointIds: Vector[CiId],
                       limit: Long, offset: Long,
                       orderBy: ExternalDeploymentOrderMode,
                       direction: ExternalDeploymentOrderDirection,
                       condition: String = ""): Vector[EndpointUpdateStatusEvent] = {
    val endpointUids = endpointIds.map(endpointId => configurationPersistence.getUid(endpointId).get).toList.asJava
    val sqlStatement =
      if (limit > 0) {
        addLimitAndOffset(
          addOrdering(
            STMT_FIND_BY_ENDPOINT_IDS,
            orderBy,
            direction
          ), limit, offset
        )
      }
      else {
        addOrdering(
          STMT_FIND_BY_ENDPOINT_IDS,
          orderBy,
          direction
        )
      }

    val result = sqlQuery(
      sqlStatement,
      params("configurationUids" -> endpointUids, "condition" -> s"%${condition}%"),
      externalDeploymentRecordRowMapper
    ).toVector
    result
  }

  private final val STMT_FIND_BY_EXTERNAL_DEPLOYMENT_UID =
    s"""${STMT_BASE_FIND}
       | WHERE conf.${CONFIGURATIONS.CI_UID} = :${CONFIGURATIONS.CI_UID}
       | AND e.${EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID} = :${EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID}
       | AND e.${EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID} = :${EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID}
       |""".stripMargin

  def findByUId(uid: ExternalDeploymentUid): Option[EndpointUpdateStatusEvent] =
    sqlQuery(
      STMT_FIND_BY_EXTERNAL_DEPLOYMENT_UID,
      params(
        CONFIGURATIONS.CI_UID -> configurationPersistence.getUid(uid.endpointId).get,
        EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID -> uid.applicationCorrelationId,
        EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID -> uid.destionationCorrelationId
      ),
      externalDeploymentRecordRowMapper
    ).headOption

  private val externalDeploymentRecordRowMapper: RowMapper[EndpointUpdateStatusEvent] = (rs: ResultSet, _: Int) =>
    EndpointUpdateStatusEvent(
      rs.getString(EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID),
      UpdateStatusEvent(
        rs.getString(EXTERNAL_DEPLOYMENT.APPLICATION_NAME),
        rs.getString(EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID),
        rs.getString(EXTERNAL_DEPLOYMENT.APPLICATION_PATH),
        rs.getString(EXTERNAL_DEPLOYMENT.STATUS_GROUP),
        StateMetadata(
          rs.getString(EXTERNAL_DEPLOYMENT.DESTINATION),
          rs.getString(EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID),
          rs.getString(EXTERNAL_DEPLOYMENT.NAMESPACE),
          VersionState(rs.getString(EXTERNAL_DEPLOYMENT.VERSION_TAG), rs.getString(EXTERNAL_DEPLOYMENT.VERSION_STATE)),
          rs.getString(EXTERNAL_DEPLOYMENT.STATUS),
          rs.getString(EXTERNAL_DEPLOYMENT.DEPLOYMENT_TYPE),
          rs.getString(EXTERNAL_DEPLOYMENT.USER),
          Option(rs.getTimestamp(EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME)).map(ts => new Date(ts.getTime)).orNull
        )
      )
    )

  private final val STMT_INSERT_APPLICATION_STATUS =
    s"INSERT INTO ${EXTERNAL_DEPLOYMENT.TABLE} " +
      s"VALUES(:${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID}, " +
      s":${EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID}, " +
      s":${EXTERNAL_DEPLOYMENT.APPLICATION_NAME}, " +
      s":${EXTERNAL_DEPLOYMENT.APPLICATION_PATH}, " +
      s":${EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID}, " +
      s":${EXTERNAL_DEPLOYMENT.DESTINATION}, " +
      s":${EXTERNAL_DEPLOYMENT.NAMESPACE}, " +
      s":${EXTERNAL_DEPLOYMENT.STATUS_GROUP}, " +
      s":${EXTERNAL_DEPLOYMENT.STATUS}, " +
      s":${EXTERNAL_DEPLOYMENT.VERSION_TAG}, " +
      s":${EXTERNAL_DEPLOYMENT.VERSION_STATE}, " +
      s":${EXTERNAL_DEPLOYMENT.DEPLOYMENT_TYPE}, " +
      s":${EXTERNAL_DEPLOYMENT.USER}, " +
      s":${EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME})"

  protected def insertEndpointPath(uid: ExternalDeploymentUid, deployment: ExternalDeployment): Unit =
    sqlExec(STMT_INSERT_APPLICATION_STATUS,
      externalDeploymentParams(uid, deployment),
      _.execute()
    )

  private final val STMT_UPDATE_APPLICATION_STATUS =
    s"""UPDATE ${EXTERNAL_DEPLOYMENT.TABLE}
       | SET
       | ${EXTERNAL_DEPLOYMENT.APPLICATION_NAME} = :${EXTERNAL_DEPLOYMENT.APPLICATION_NAME},
       | ${EXTERNAL_DEPLOYMENT.APPLICATION_PATH} = :${EXTERNAL_DEPLOYMENT.APPLICATION_PATH},
       | ${EXTERNAL_DEPLOYMENT.DESTINATION} = :${EXTERNAL_DEPLOYMENT.DESTINATION},
       | ${EXTERNAL_DEPLOYMENT.NAMESPACE} = :${EXTERNAL_DEPLOYMENT.NAMESPACE},
       | ${EXTERNAL_DEPLOYMENT.STATUS_GROUP} = :${EXTERNAL_DEPLOYMENT.STATUS_GROUP},
       | ${EXTERNAL_DEPLOYMENT.STATUS} = :${EXTERNAL_DEPLOYMENT.STATUS},
       | ${EXTERNAL_DEPLOYMENT.VERSION_TAG} = :${EXTERNAL_DEPLOYMENT.VERSION_TAG},
       | ${EXTERNAL_DEPLOYMENT.VERSION_STATE} = :${EXTERNAL_DEPLOYMENT.VERSION_STATE},
       | ${EXTERNAL_DEPLOYMENT.DEPLOYMENT_TYPE} = :${EXTERNAL_DEPLOYMENT.DEPLOYMENT_TYPE},
       | ${EXTERNAL_DEPLOYMENT.USER} = :${EXTERNAL_DEPLOYMENT.USER},
       | ${EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME} = :${EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME}
       | WHERE ${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID} = :${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID}
       | AND ${EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID} = :${EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID}
       | AND ${EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID} = :${EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID}
       |""".stripMargin

  protected def updateEndpointPath(uid: ExternalDeploymentUid, deployment: ExternalDeployment): Unit = {
    sqlExec(
      STMT_UPDATE_APPLICATION_STATUS,
      externalDeploymentParams(uid, deployment),
      _.execute()
    )
  }

  private def externalDeploymentParams(uid: ExternalDeploymentUid, deployment: ExternalDeployment) = {
    params(
      EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID -> configurationPersistence.getUid(uid.endpointId).get,
      EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID -> uid.applicationCorrelationId,
      EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID -> uid.destionationCorrelationId,
      EXTERNAL_DEPLOYMENT.APPLICATION_NAME -> deployment.applicationName,
      EXTERNAL_DEPLOYMENT.APPLICATION_PATH -> deployment.applicationPath,
      EXTERNAL_DEPLOYMENT.DESTINATION -> deployment.destination,
      EXTERNAL_DEPLOYMENT.NAMESPACE -> deployment.namespace,
      EXTERNAL_DEPLOYMENT.STATUS_GROUP -> deployment.statusGroup,
      EXTERNAL_DEPLOYMENT.STATUS -> deployment.status,
      EXTERNAL_DEPLOYMENT.VERSION_TAG -> deployment.versionTag,
      EXTERNAL_DEPLOYMENT.VERSION_STATE -> deployment.versionState,
      EXTERNAL_DEPLOYMENT.DEPLOYMENT_TYPE -> deployment.deploymentType,
      EXTERNAL_DEPLOYMENT.USER -> deployment.user,
      EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME -> deployment.lastChangeTime
    )
  }

  private val STMT_DELETE_FOR_WEBHOOK_CONF_UID =
    s"""
       |DELETE FROM ${EXTERNAL_DEPLOYMENT.TABLE}
       |WHERE ${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID} = :${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID}
     """.stripMargin

  def delete(endpointId: CiId): Boolean = {
    sqlExec(STMT_DELETE_FOR_WEBHOOK_CONF_UID, params(EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID ->
      configurationPersistence.getUid(endpointId).get), _.execute())
  }
4

  private val STMT_DELETE_FOR_EXTERNAL_DEPLOYMENT_UID =
    s"""
       |DELETE FROM ${EXTERNAL_DEPLOYMENT.TABLE}
       |WHERE ${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID} = :${EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID} AND
       |${EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID} = :${EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID} AND
       |${EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID} = :${EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID}
     """.stripMargin
  protected def delete(uid: ExternalDeploymentUid): Unit = {
    sqlExec(STMT_DELETE_FOR_EXTERNAL_DEPLOYMENT_UID,
      params(
        EXTERNAL_DEPLOYMENT.WEBHOOK_CONFIGURATION_UID -> configurationPersistence.getUid(uid.endpointId).get,
        EXTERNAL_DEPLOYMENT.APPLICATION_CORRELATION_UID -> uid.applicationCorrelationId,
        EXTERNAL_DEPLOYMENT.DESTINATION_CORRELATION_UID -> uid.destionationCorrelationId
      ),
      _.execute())
  }

  private def addOrdering(query: String, order: ExternalDeploymentOrderMode, direction: ExternalDeploymentOrderDirection): String =
    getOrderCriterionColumn(order).mkString(
      s"${query} ORDER BY ",
      s" ${getOrderCriterionDirection(direction)}, ",
      s" ${getOrderCriterionDirection(direction)}")

  private def getOrderCriterionDirection(direction: ExternalDeploymentOrderDirection): String =
    direction match {
      case ExternalDeploymentOrderDirection.ASC => "ASC"
      case ExternalDeploymentOrderDirection.DESC => "DESC"
    }

  private def getOrderCriterionColumn(order: ExternalDeploymentOrderMode): List[String] =
    order match {
      case ExternalDeploymentOrderMode.APPLICATION =>
        List(EXTERNAL_DEPLOYMENT.APPLICATION_NAME,EXTERNAL_DEPLOYMENT.VERSION_TAG, EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME)
      case ExternalDeploymentOrderMode.DESTINATION =>
        List(EXTERNAL_DEPLOYMENT.DESTINATION, EXTERNAL_DEPLOYMENT.NAMESPACE, EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME)
      case ExternalDeploymentOrderMode.VERSION =>
        List(EXTERNAL_DEPLOYMENT.VERSION_TAG, EXTERNAL_DEPLOYMENT.APPLICATION_NAME, EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME)
      case ExternalDeploymentOrderMode.STATUS =>
        List(EXTERNAL_DEPLOYMENT.STATUS_GROUP, EXTERNAL_DEPLOYMENT.STATUS, EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME)
      case ExternalDeploymentOrderMode.TIME =>
        List(EXTERNAL_DEPLOYMENT.LAST_EVENT_TIME)
    }
}

case class ExternalDeploymentUid(endpointId: CiId, applicationCorrelationId: String, destionationCorrelationId: String)

case class ExternalDeployment(
                               applicationName: String,
                               applicationPath: String,
                               destination: String,
                               namespace: String,
                               statusGroup: String,
                               status: String,
                               versionTag: String,
                               versionState: String,
                               deploymentType: String,
                               user: String,
                               lastChangeTime: Date
                             )

object ExternalDeployment {
  def apply(event: UpdateStatusEvent): ExternalDeployment = {
    val (deploymentStatus, deploymentType, user, versionLabel, versionState, time) =
      Option(event.state).map(e => {
        val (versionState, versionLabel) = Option(e.versionTag).map(v => (v.state, v.label)).getOrElse((null, null))
        (e.deploymentStatus,
          e.deploymentType,
          e.user,
          versionLabel,
          versionState,
          e.lastChangeTime
        )
      }).getOrElse((null, null, null, null, null, null))
    ExternalDeployment(
      event.applicationName,
      event.applicationPath,
      event.destination,
      event.namespace,
      event.statusGroup,
      deploymentStatus,
      versionLabel,
      versionState,
      deploymentType,
      user,
      time
    )
  }

}
