package com.xebialabs.xlrelease.versioning.templates

import com.xebialabs.deployit.plumbing.serialization.ResolutionContext
import com.xebialabs.deployit.server.api.upgrade.Version
import com.xebialabs.xlplatform.upgrade.RepositoryVersionService
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.events.ReleaseRestoredEvent
import com.xebialabs.xlrelease.domain.{Release, ReleaseTrigger}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository.{DependencyTargetResolver, Ids, ReleaseRepository}
import com.xebialabs.xlrelease.service.ReleaseService
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.utils.CiHelper
import com.xebialabs.xlrelease.versioning.TemplateVersioningException
import com.xebialabs.xlrelease.versioning.templates.domain.events.{RevisionDeletedEvent, VersionCreatedEvent}
import com.xebialabs.xlrelease.versioning.templates.repository.persistence.data.{TemplateRevision, TemplateRevisionData, TemplateRevisionType}
import com.xebialabs.xlrelease.versioning.templates.repository.{TemplateRevisionDataRepository, TemplateRevisionRepository}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.Primary
import org.springframework.stereotype.Service

import java.util
import java.util.Date
import java.util.stream.Collectors
import scala.util.{Failure, Try}

@Service
@Primary
@IsTransactional
class DbTemplateStorageService @Autowired()(templateRevisionRepository: TemplateRevisionRepository,
                                            templateRevisionDataRepository: TemplateRevisionDataRepository,
                                            releaseVersioningSerialization: ReleaseVersioningSerialization,
                                            releaseService: ReleaseService,
                                            releaseRepository: ReleaseRepository,
                                            repositoryVersionService: RepositoryVersionService,
                                            eventBus: XLReleaseEventBus,
                                            dependencyTargetResolver: DependencyTargetResolver
                                           )
  extends TemplateStorageService with Logging {

  override def saveRevision(template: Release, user: String, message: String): Unit = {
    createRevision(template, user, message)
  }

  override def saveVersion(templateId: String, user: String, versionName: String, message: String): Try[Int] = {
    val template = releaseService.findById(templateId)
    Try {
      val version = createRevision(template, user, message, versionName)
      templateRevisionRepository.deleteRevisionsForVersion(version)
      eventBus.publish(VersionCreatedEvent(templateId, versionName))
      version.id
    }.recoverWith {
      case e: Exception =>
        logger.error("Unable to create new template version", e)
        Failure(e)
    }
  }

  private def createRevision(template: Release, user: String, description: String, version: String = null): TemplateRevision = {
    if (template.isWorkflow) {
      throw TemplateVersioningException("Template version control is not supported for workflows")
    }

    val templateRevision = new TemplateRevision()
    templateRevision.ciUid = template.getCiUid
    templateRevision.revisionType = if (version != null) TemplateRevisionType.VERSION else TemplateRevisionType.REVISION
    templateRevision.version = version
    templateRevision.description = description
    templateRevision.username = user
    templateRevision.revisionTime = new Date()

    val savedTr = templateRevisionRepository.create(templateRevision)

    val trd = TemplateRevisionData(savedTr.id, xlrVersion, releaseVersioningSerialization.toBytes(template))
    templateRevisionDataRepository.create(trd)
    savedTr
  }

  override def restore(templateId: String, revisionId: Int): Release = {
    val currentTemplate = releaseService.findById(templateId)

    val context = if (Ids.isInFolder(templateId)) {
      ResolutionContext(Ids.getParentId(templateId))
    } else {
      ResolutionContext.GLOBAL
    }

    val previousTemplate = load(revisionId, context)

    processRestoredTemplate(previousTemplate, currentTemplate)

    val updatedTemplate = releaseRepository.replace(currentTemplate, previousTemplate)

    eventBus.publish(ReleaseRestoredEvent(currentTemplate, updatedTemplate, revisionId))

    releaseService.findById(templateId)
  }

  override def load(revisionId: Int, context: ResolutionContext): Release = {
    templateRevisionDataRepository.findById(revisionId) match {
      case Some(trd) =>
        releaseVersioningSerialization.fromBytes(trd.content, trd.dataModelVersion, context) match {
          case Some(template) => processDependencies(template)
          case None => throw new LogFriendlyNotFoundException(s"Error reading $revisionId, see logs for more details")
        }
      case None => throw new LogFriendlyNotFoundException(s"Revision id $revisionId not found")
    }
  }

  override def deleteRevision(templateId: String, revisionId: Int): Unit = {
    releaseRepository.getUid(templateId).foreach(templateCiUid => {
      templateRevisionRepository.deleteRevisions(templateCiUid, Seq(revisionId))
      eventBus.publish(RevisionDeletedEvent(templateId, revisionId))
    })
  }

  override def deleteRevisions(templateId: String, revisionIds: Seq[Int]): Unit = {
    releaseRepository.getUid(templateId).foreach(templateCiUid => {
      templateRevisionRepository.deleteRevisions(templateCiUid, revisionIds)
      revisionIds.foreach(id => eventBus.publish(RevisionDeletedEvent(templateId, id)))
    })
  }

  private def processRestoredTemplate(restoredTemplate: Release, currentTemplate: Release): Unit = {
    restoredTemplate.setCiUid(currentTemplate.getCiUid)
    CiHelper.rewriteWithNewId(restoredTemplate, currentTemplate.getId)
    restoredTemplate.setReleaseTriggers(new util.ArrayList[ReleaseTrigger]())
    processDependencies(restoredTemplate)
    restoredTemplate.get$ciAttributes().setScmTraceabilityDataId(null)
  }

  private def processDependencies(restoredTemplate: Release): Release = {
    restoredTemplate.getAllGates.forEach(gateTask => {
      val validDependencies = gateTask.getDependencies.stream().filter(d =>
        d.hasResolvedTarget || Try(dependencyTargetResolver.resolveTarget(d)).isSuccess
      ).collect(Collectors.toList())

      gateTask.setDependencies(validDependencies)
    })

    restoredTemplate
  }

  private def xlrVersion: String = {
    val version = repositoryVersionService.readVersionOfComponent(XL_RELEASE_COMPONENT)
    if (null == version) {
      Version.VERSION_0
    } else {
      version.getVersion
    }
  }
}
