package com.xebialabs.xlrelease.versioning.templates

import com.xebialabs.xlrelease.activity._
import com.xebialabs.xlrelease.domain.events.XLReleaseEvent
import com.xebialabs.xlrelease.events._
import com.xebialabs.xlrelease.service.ReleaseService
import com.xebialabs.xlrelease.user.User
import com.xebialabs.xlrelease.versioning.templates.configuration.domain.VersioningSettings
import com.xebialabs.xlrelease.versioning.templates.domain.events.TemplateVersioningEvent
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

@Component
@EventListener
class TemplateChangeListener @Autowired()(val templateStorage: TemplateStorageService,
                                          releaseService: ReleaseService,
                                          logEntryFactory: DefaultLogEntryFactory)
  extends Logging {

  @Subscribe
  def onEvent(event: XLReleaseEvent): Unit = {
    if (VersioningSettings.automaticChangesEnabled) {
      event match {
        case _: TemplateVersioningEvent => ()
        case _ =>
          val loggingParams = logEntryFactory.getLogEntries(event)
          loggingParams match {
            case LoggingParams(Some(releaseId), user, logEntries) if releaseService.isTemplate(releaseId) && logEntries.nonEmpty =>
              val template = releaseService.findById(releaseId)
              if (!template.isWorkflow) {
                val msg = logEntries.map(_.getMessage).mkString(",")
                val revisionAuthor: String = user.orElse(Some(User.AUTHENTICATED_USER.getName)).orNull
                templateStorage.saveRevision(template, revisionAuthor, msg)
              }
            case _ =>
              logger.debug(s"Unable to create revision based on event $event")
          }
      }
    }
  }
}

@Component
class DefaultLogEntryFactory @Autowired()(logEntryFactories: Array[ActivityLogEntryFactory])
  extends Logging {

  private lazy val combineFactories = logEntryFactories.map(_.logEntries).reduceLeft(_ orElse _) orElse ignored

  def getLogEntries(e: XLReleaseEvent): LoggingParams = {
    combineFactories(e)
  }

  def ignored: PartialFunction[XLReleaseEvent, LoggingParams] = {
    case default =>
      logger.debug(s"Event $default not traced by one of the activity log entry factories")
      LoggingParams(None, None, List())
  }
}
