package com.xebialabs.xlrelease.domain;

import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.Set;
import com.google.common.collect.ImmutableMap;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.xlrelease.activity.ActivityOps;
import com.xebialabs.xlrelease.triggers.events.TriggerEvent;

import static com.google.common.collect.Sets.newLinkedHashSet;
import static com.xebialabs.xlrelease.domain.ActivityCategory.EXECUTION;
import static com.xebialabs.xlrelease.domain.ActivityCategory.LIFECYCLE;
import static com.xebialabs.xlrelease.domain.ActivityCategory.TRIGGER_EDIT;

public enum TriggerActivity implements ActivityOps {
    TRIGGER_CREATED("Created trigger '%s'", TRIGGER_EDIT),
    TRIGGER_PROPERTY_UPDATED("Changed property '%s' from '%s' to '%s'", TRIGGER_EDIT),
    TRIGGER_CREATED_FROM_AS_CODE("Created trigger from As-code %s", TRIGGER_EDIT),
    TRIGGER_PROPERTY_UPDATED_FROM_AS_CODE("Updated trigger from As-code %s", TRIGGER_EDIT),
    TRIGGER_DELETED("Deleted trigger '%s'", TRIGGER_EDIT),

    TRIGGER_ENABLED("Enabled trigger '%s'", LIFECYCLE),
    TRIGGER_DISABLED("Disabled trigger '%s'", LIFECYCLE),
    TRIGGER_AUTO_DISABLED("Disabled trigger '%s' automatically due to reaching the maximum configured number of consecutive failures.", LIFECYCLE),
    TRIGGER_EXECUTED("%s from trigger '%s'", EXECUTION),
    TRIGGER_SKIPPED("Skipped executing trigger '%s'. Reason: %s", EXECUTION),
    TRIGGER_FAILED("Failed trigger '%s' due to error: %s", EXECUTION),
    TRIGGER_FILTER_FAILED("Failed trigger '%s' filter due to error: %s", EXECUTION),
    OTHER("", ActivityCategory.OTHER);

    private String message;
    private Set<ActivityCategory> categories;

    TriggerActivity(String message, ActivityCategory... categories) {
        this.message = message;
        Set<ActivityCategory> myCategories = newLinkedHashSet();
        if (categories != null) {
            Collections.addAll(myCategories, categories);
        }
        this.categories = Collections.unmodifiableSet(myCategories);
    }

    public ActivityLogEntry create(Date eventTime, String username, Type targetType, String targetId, Object... parameters) {
        return create(eventTime, name(), username, targetType, targetId, message, parameters);
    }

    public ActivityLogEntry create(TriggerEvent ev, BaseConfigurationItem target, Object... parameters) {
        return create(ev.timestamp(), ev.username(), target.getType(), target.getId(), parameters);
    }

    @Override
    public String getName() {
        return name();
    }

    public Set<ActivityCategory> getCategories() {
        return categories;
    }

    private static final Map<String, TriggerActivity> BY_NAME;

    static {
        ImmutableMap.Builder<String, TriggerActivity> builder = ImmutableMap.builder();
        for (TriggerActivity value : TriggerActivity.values()) {
            builder.put(value.name(), value);
        }
        BY_NAME = builder.build();
    }

    public static TriggerActivity safeValueOf(String name) {
        TriggerActivity activity = BY_NAME.get(name);
        return (activity == null ? OTHER : activity);
    }
}
