package com.xebialabs.xlrelease.triggers.service

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlplatform.webhooks.domain.HttpRequestEvent
import com.xebialabs.xlplatform.webhooks.events.handlers.EventConsumerHandler
import com.xebialabs.xlrelease.script.ScriptVariables
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.triggers.action.CreateReleaseFromTemplateAction
import com.xebialabs.xlrelease.triggers.api.view.{CreateReleaseFromTemplateActionView, EventBasedTriggerTestRunResult}
import com.xebialabs.xlrelease.triggers.event_based.{EventBasedTrigger, EventBasedTriggerConsumer, EventFiltering}
import com.xebialabs.xlrelease.triggers.repository.TriggerPersistenceHelper
import com.xebialabs.xlrelease.webhooks.mapping.PropertiesMapper
import com.xebialabs.xlrelease.webhooks.untyped.EventContentParser
import grizzled.slf4j.{Logger, Logging}
import org.slf4j.helpers.NOPLogger
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.ApplicationContext
import org.springframework.stereotype.Component

import javax.script.ScriptException

@Component
class EventTriggerTestRunService @Autowired()(applicationContext: ApplicationContext,
                                              ciIdService: CiIdService,
                                              implicit val scriptVariables: ScriptVariables,
                                              implicit val eventParser: EventContentParser) {

  private lazy val nopLogger = new Logger(NOPLogger.NOP_LOGGER)

  def testRun(trigger: EventBasedTrigger, event: HttpRequestEvent): EventBasedTriggerTestRunResult = {
    trigger.setTriggerActionType(Type.valueOf(classOf[CreateReleaseFromTemplateAction]).toString)
    trigger.setId(s"${trigger.getFolderId}/TestRunTriggerId")
    TriggerPersistenceHelper.fixPropertyIds(trigger, ciIdService)

    val result = new EventBasedTriggerTestRunResult
    try {
      result.filterResult = doFilter(trigger, event)
    } catch {
      case e@(_: RuntimeException | _: ScriptException) => result.filterError = e.getMessage
    }
    try {
      result.mappedProperties = doMapper(trigger, event)
    } catch {
      case e@(_: RuntimeException | _: ScriptException) => result.mappedPropertiesError = e.getMessage
    }

    result
  }

  private def doFilter(trigger: EventBasedTrigger, event: HttpRequestEvent) = {
    val testConsumerHandler = new TestRunEventBasedTriggerConsumerHandler(trigger, applicationContext)
    testConsumerHandler.filter(event)
  }

  private def doMapper(trigger: EventBasedTrigger, event: HttpRequestEvent) = {
    val action = trigger.instantiateAction().asInstanceOf[CreateReleaseFromTemplateAction]
    applicationContext.getAutowireCapableBeanFactory.autowireBean(action)
    action.init(trigger)
    // map properties event -> action
    new PropertiesMapperWithoutLogging().mapProperties(trigger, event, action)

    val view = new CreateReleaseFromTemplateActionView
    view.templateId = action.templateId
    view.releaseTitle = action.releaseTitle
    view.tags = action.getReleaseTags
    view.variables = action.variables
    view
  }

  private class TestRunEventBasedTriggerConsumerHandler(trigger: EventBasedTrigger, applicationContext: ApplicationContext)
    extends EventConsumerHandler[HttpRequestEvent, EventBasedTriggerConsumer]
      with Logging
      with EventFiltering {

    override def logger: Logger = nopLogger

    override def filter(config: EventBasedTriggerConsumer, event: HttpRequestEvent): Boolean = {
      doFilter(trigger, event, applicationContext)
    }

    override def consumeEvent(config: EventBasedTriggerConsumer, event: HttpRequestEvent): Boolean = true

    def filter(event: HttpRequestEvent): Boolean = {
      filter(new EventBasedTriggerConsumer(trigger), event)
    }
  }

  private class PropertiesMapperWithoutLogging extends PropertiesMapper {
    override def logger: Logger = nopLogger
  }

}
