package com.xebialabs.xlrelease.triggers.actors

import com.google.common.base.Objects.equal
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage
import com.xebialabs.xlrelease.builder.ReleaseBuilder
import com.xebialabs.xlrelease.domain.{ReleaseTrigger, Trigger}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.Ids.formatWithFolderId
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.service.ConfigurationVariableService
import com.xebialabs.xlrelease.triggers.action.{TriggerActionExecuted, TriggerActionSkipped}
import com.xebialabs.xlrelease.triggers.events._
import com.xebialabs.xlrelease.triggers.scheduled.ScheduledTriggerExecutionContext
import com.xebialabs.xlrelease.triggers.service.impl.{TriggerExecutionContext, TriggerExecutionResult, TriggerLifecycle}
import com.xebialabs.xlrelease.utils.CiHelper.fixUpInternalReferences
import com.xebialabs.xlrelease.utils.FolderId
import com.xebialabs.xlrelease.utils.PasswordVerificationUtils.replacePasswordPropertiesInCiIfNeeded
import com.xebialabs.xlrelease.validation.XlrValidationsFailedException
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success}

class TriggerOperations(triggerRepository: TriggerRepository,
                        releaseRepository: ReleaseRepository,
                        eventBus: XLReleaseEventBus,
                        triggerLifecycle: TriggerLifecycle[Trigger],
                        configurationVariableService: ConfigurationVariableService)
  extends Logging {


  def find(triggerId: String): Trigger = {
    triggerRepository.find(triggerId)
  }

  def addTrigger(trigger: Trigger): Trigger = {
    validateFolderId(trigger)
    replacePasswordPropertiesInCiIfNeeded(None, trigger)
    checkValidation(trigger, true)
    updateVariables(trigger)
    fixUpInternalReferences(trigger)
    eventBus.publishAndFailOnError(TriggerCreatingActionEvent(trigger))
    val addedTrigger = triggerRepository.create(trigger)
    eventBus.publish(TriggerCreatedEvent(addedTrigger))
    enableTrigger(trigger)
    addedTrigger
  }

  /**
   * Updates trigger or only selected properties of a trigger and optionally emits events.
   * If there are no events emitted validation will NOT be invoked.
   *
   * @param trigger    trigger object with new values
   * @param properties properties that will be updated. If none are provided whole trigger object will be replaced.
   * @param emitEvents if events should be emitted
   * @return updated trigger object
   */
  @Timed
  def updateTrigger(original: Trigger,
                    trigger: Trigger,
                    validate: Boolean,
                    emitEvents: Boolean,
                    internal: Boolean,
                    checkReferencePermissions: Boolean,
                    properties: Seq[String] = Seq(),
                   ): Trigger = {
    import com.xebialabs.xlrelease.triggers._
    trace(
      s"""Updating trigger $original properties: '${properties.mkString(",")}'
         |previous: ${original.toJson}
         |new: ${trigger.toJson}""".stripMargin
    )

    val toUpdate = if (properties.isEmpty) {
      for (internalProperty <- trigger.getInternalProperties.asScala) {
        trigger.setProperty(internalProperty, original.getProperty(internalProperty))
      }
      trigger
    } else {
      CiCloneHelper.cloneCi(original)
    }
    toUpdate.setCiUid(original.getCiUid)
    toUpdate.setFolderId(FolderId(trigger.getFolderId).absolute)
    toUpdate.setId(formatWithFolderId(trigger.getFolderId, Ids.getName(trigger.getId)))
    fixUpInternalReferences(toUpdate)

    if (!FolderId(original.getFolderId).absolute.equals(Ids.ROOT_FOLDER_ID)) {
      validateFolderId(toUpdate)
    }

    for (propertyToUpdate <- properties) {
      toUpdate.setProperty(propertyToUpdate, trigger.getProperty(propertyToUpdate))
    }

    if (validate) {
      replacePasswordPropertiesInCiIfNeeded(Some(original), toUpdate)
      checkValidation(toUpdate, checkReferencePermissions)
      updateVariables(toUpdate)
      eventBus.publishAndFailOnError(TriggerUpdatingActionEvent(original, toUpdate))
    }
    val updated = triggerRepository.update(toUpdate)
    if (!internal) {
      val emitStateEvent = emitEvents && (!equal(original.isEnabled, updated.isEnabled) || properties.contains("enabled")) // don't ask...
      enableTrigger(updated, emitStateEvent)
    }
    if (emitEvents) {
      eventBus.publish(TriggerUpdatedEvent(original, updated))
    }
    updated
  }

  @Timed
  def deleteTrigger(trigger: Trigger): Unit = {
    trigger.setEnabled(false)
    eventBus.publishAndFailOnError(TriggerDeletingActionEvent(trigger))
    triggerRepository.delete(trigger.getId)
    enableTrigger(trigger)
    eventBus.publish(TriggerDeletedEvent(trigger))
  }

  @Timed
  def execute(trigger: Trigger, executionContext: TriggerExecutionContext): Trigger = {
    configurationVariableService.resolveFromCi(trigger)()
    val TriggerExecutionResult(maybeExecutedTrigger, result) = triggerLifecycle.execute(trigger, executionContext)
    result match {
      case Failure(e) =>
        eventBus.publish(TriggerFailedEvent(trigger, e.getMessage, executionContext.getExecutionDataId))
        throw e
      case Success(actionResult) =>
        actionResult match {
          case TriggerActionExecuted(result) =>
            eventBus.publish(TriggerExecutedEvent(maybeExecutedTrigger, result, executionContext.getExecutionDataId))
          case TriggerActionSkipped(message) if !executionContext.isInstanceOf[ScheduledTriggerExecutionContext] =>
            eventBus.publish(TriggerSkippedEvent(maybeExecutedTrigger, message, executionContext.getExecutionDataId))
          case _ => ()
        }
        maybeExecutedTrigger
    }
  }

  private def enableTrigger(trigger: Trigger, emitEvents: Boolean = true): Unit = {
    if (trigger.isEnabled) {
      triggerLifecycle.enable(trigger, true)
      if (emitEvents) {
        eventBus.publish(TriggerEnabledEvent(trigger))
      }
    } else {
      triggerLifecycle.disable(trigger)
      if (emitEvents) {
        eventBus.publish(TriggerDisabledEvent(trigger))
      }
    }
  }

  private def validateFolderId(trigger: Trigger): Unit = {
    if (FolderId(trigger.getFolderId).absolute.equals(Ids.ROOT_FOLDER_ID)) {
      trigger.get$validationMessages().add(new ValidationMessage(trigger.getId, "folderId",
        "You cannot create trigger on the root folder. The root folder path is not supported."))
    }
  }

  private def checkValidation(trigger: Trigger, msgs: java.util.List[ValidationMessage]): Unit = {
    if (msgs.asScala.nonEmpty) {
      trigger.get$validationMessages().addAll(msgs)
    }
    if (trigger.get$validationMessages().asScala.nonEmpty) {
      throw new XlrValidationsFailedException(trigger)
    }
  }

  private def checkValidation(ci: Trigger, checkReferencePermissions: Boolean): Unit = {
    checkValidation(ci, triggerLifecycle.validate(ci, checkReferencePermissions))
  }

  private def updateVariables(trigger: Trigger): Unit = {
    trigger match {
      case t: ReleaseTrigger =>
        val variables = releaseRepository.findById(t.getTemplate, ResolveOptions.WITH_DECORATORS).getVariables.asScala.filter(v => t.getVariablesByKeys.containsKey(v.getKey)).toList.asJava
        val variableHolderTemplate = ReleaseBuilder.newRelease().withId(t.getTemplate).withVariables(variables).build()
        variableHolderTemplate.setVariableValues(t.getTemplateVariableValues(!_.isPassword).asScala.filter(_._2 != null).asJava)
        variableHolderTemplate.setPasswordVariableValues(t.getTemplatePasswordVariables)
        t.setVariables(variableHolderTemplate.getVariables.asScala.filter(_.getId != null).toList.asJava)
      case _ =>
    }
  }

}
