package com.xebialabs.xlrelease.triggers.scheduled.quartz

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.configuration.TriggerDataPurgeSettings
import com.xebialabs.xlrelease.configuration.TriggerDataPurgeSettings.TRIGGER_DATA_PURGE_SETTINGS_ID
import com.xebialabs.xlrelease.domain.events.ConfigurationUpdatedEvent
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.quartz.release.scheduler.{QuartzScheduledService, ReleaseSchedulerService}
import com.xebialabs.xlrelease.service.ConfigurationService
import com.xebialabs.xlrelease.triggers.scheduled.quartz.TriggerDataPurgeJob.DATA_PURGE_AGE_KEY
import com.xebialabs.xlrelease.triggers.scheduled.quartz.TriggerDataPurgeService.{TRIGGER_DATA_PURGE_SERVICE, triggerPurgeDataSettingsType}
import org.quartz.JobBuilder.newJob
import org.quartz.SimpleScheduleBuilder.simpleSchedule
import org.quartz.TriggerBuilder.newTrigger
import org.quartz.{JobDataMap, JobDetail, SimpleTrigger, Trigger}
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

object TriggerDataPurgeService {
  val TRIGGER_DATA_PURGE_SERVICE = "Triggers data purge"
  val triggerPurgeDataSettingsType = Type.valueOf(classOf[TriggerDataPurgeSettings])
}

@Service
@EventListener
class TriggerDataPurgeService(val releaseSchedulerService: ReleaseSchedulerService,
                              val configuration: ConfigurationService)
  extends QuartzScheduledService {

  val serviceName: String = TRIGGER_DATA_PURGE_SERVICE
  private val jobClazz = classOf[TriggerDataPurgeJob]
  private val triggerDataPurgeJobInterval = XlrConfig.getInstance.durations.triggerDataPurgeJobInterval

  private val jobName: String = s"$serviceName JOB".toUpperCase.split(' ').mkString("_")
  private val groupName: String = serviceName.toUpperCase.split(' ').mkString("_")

  lazy val job: JobDetail = newJob(jobClazz)
    .withDescription(s"$serviceName job")
    .withIdentity(jobName, groupName)
    .storeDurably(true)
    .setJobData(
      new JobDataMap(
        Map[String, String](DATA_PURGE_AGE_KEY -> getTriggerPurgeDataSettings.getTriggersDataPurgeAgeThreshold.toString).asJava
      )
    )
    .build()

  lazy val trigger: SimpleTrigger = newTrigger()
    .withIdentity(jobName, groupName)
    .withDescription(s"Trigger $serviceName jobs")
    .withSchedule(
      simpleSchedule().withIntervalInMilliseconds(triggerDataPurgeJobInterval.toMillis).repeatForever()
        .withMisfireHandlingInstructionNowWithRemainingCount()
    )
    .startNow()
    .build()


  private def getTriggerPurgeDataSettings: TriggerDataPurgeSettings = {
    configuration.read(TRIGGER_DATA_PURGE_SETTINGS_ID).asInstanceOf[TriggerDataPurgeSettings]
  }

  @AsyncSubscribe
  def onConfigurationUpdate(event: ConfigurationUpdatedEvent): Unit = {
    if (event.updated.getType.instanceOf(triggerPurgeDataSettingsType)) {
      scheduleOrUnschedule()
    }
  }

  private def scheduleOrUnschedule(): Unit = {
    try {
      if (getTriggerPurgeDataSettings.isEnabled) {
        job.getJobDataMap.put(DATA_PURGE_AGE_KEY, getTriggerPurgeDataSettings.getTriggersDataPurgeAgeThreshold.toString)
        releaseSchedulerService.schedule(job, trigger)
      } else {
        releaseSchedulerService.unschedule(job.getKey)
      }
    } catch {
      case e: Exception => logger.error(s"Error while starting $serviceName service", e)
    }
  }

  override def getJobDetail: JobDetail = job

  override def getTrigger: Trigger = trigger

  override def isEnabled: Boolean = getTriggerPurgeDataSettings.isEnabled
}
