package com.xebialabs.xlrelease.triggers.activity

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind._
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.domain.variables._
import com.xebialabs.xlrelease.domain.{ActivityLogEntry, Trigger, TriggerActivity}
import com.xebialabs.xlrelease.triggers.event_based.EventFilter
import com.xebialabs.xlrelease.webhooks.mapping.MappedProperty
import com.xebialabs.xlrelease.webhooks.mapping.MappedProperty.PasswordValue

import java.util.Date
import java.{lang, util}
import scala.jdk.CollectionConverters._

case class ChangedProperty(name: String, oldValue: Any, newValue: Any)

object TriggerFieldsComparator {
  val MASK: String = "*****"

  def diffListsOf[T](original: Seq[T], updated: Seq[T],
                     keyFn: T => String,
                     labelFn: T => String,
                     valueFn: T => Any,
                     maskFn: T => Boolean = (_: T) => false): Seq[ChangedProperty] = {
    val allKeys = (original ++ updated).map(keyFn).distinct

    allKeys.flatMap { key =>
      val oldItem = original.find(item => keyFn(item) == key)
      val newItem = updated.find(item => keyFn(item) == key)

      val oldValue = oldItem.map(valueFn).getOrElse("")
      val newValue = newItem.map(valueFn).getOrElse("")

      if (oldValue != newValue) {
        val label = oldItem.orElse(newItem).map(labelFn).get
        if (newItem.exists(maskFn)) {
          Seq(ChangedProperty(label, MASK, MASK))
        } else {
          Seq(ChangedProperty(label, oldValue, newValue))
        }
      } else {
        Seq.empty
      }
    }
  }

  //noinspection ScalaStyle
  def format(untypedValue: Any): String = untypedValue match {
    case value: String => value
    case value: ConfigurationItem =>
      val ciId = value.getId
      val ciTitle = if (value.hasProperty("title")) s" (${value.getProperty("title")})" else ""
      s"$ciId$ciTitle"
    case value: util.List[_] => asVariable(classOf[ListStringVariable], value)
    case value: util.Set[_] => asVariable(classOf[SetStringVariable], value)
    case value: util.Map[_, _] => asVariable(classOf[MapStringStringVariable], value)
    case value: lang.Boolean => asVariable(classOf[BooleanVariable], value)
    case value: util.Date => asVariable(classOf[DateVariable], value)
    case value: lang.Integer => asVariable(classOf[IntegerVariable], value)
    case value => String.valueOf(value)
  }

  private def asVariable[T <: Variable](clazz: Class[T], value: Any): String = {
    val variable = clazz.newInstance()
    variable.setUntypedValue(value)
    variable.getValueAsString
  }

}

class TriggerFieldsComparator(timestamp: Date, username: String, original: Trigger, updated: Trigger) {

  import TriggerFieldsComparator._

  def getLogs: List[ActivityLogEntry] = {
    val changes = calculateTriggerChanges(original, updated)

    changes.foldLeft(List.empty[ActivityLogEntry]) { case (logs, ChangedProperty(propertyName, oldValue, newValue)) =>
      val oldValueAsString = format(oldValue)
      val newValueAsString = format(newValue)
      if (oldValueAsString != newValueAsString || oldValueAsString == MASK) { // AbstractStringView equals is flawed
        logs :+ TriggerActivity.TRIGGER_PROPERTY_UPDATED.create(
          timestamp, username, updated.getType, updated.getId, propertyName, oldValueAsString, newValueAsString)
      } else {
        logs
      }
    }
  }

  //noinspection ScalaStyle
  private def calculateTriggerChanges(original: Trigger, updated: Trigger): List[ChangedProperty] = {
    val propertyDescriptors = original.getType.getDescriptor.getPropertyDescriptors.asScala

    propertyDescriptors.flatMap { pd: PropertyDescriptor =>
      val oldValue = original.getProperty[Any](pd.getName)
      val newValue = updated.getProperty[Any](pd.getName)

      if (oldValue == newValue && pd.getName != "variables") {
        Seq.empty
      } else {
        pd.getKind match {
          case BOOLEAN if pd.getName == "enabled" => Seq.empty
          case CI if pd.getName == "eventFilter" =>
            val oldFilter = Option(oldValue.asInstanceOf[EventFilter]).map(_.formattedValue).getOrElse("")
            val newFilter = Option(newValue.asInstanceOf[EventFilter]).map(_.formattedValue).getOrElse("")
            if (oldFilter != newFilter) {
              Seq(ChangedProperty(pd.getLabel, oldFilter, newFilter))
            } else {
              Seq.empty
            }
          case LIST_OF_CI if pd.getName == "variables" =>
            diffListsOf[Variable](
              oldValue.asInstanceOf[util.List[Variable]].asScala.toSeq,
              newValue.asInstanceOf[util.List[Variable]].asScala.toSeq,
              item => item.getKey,
              item => Option(item.getLabel).getOrElse(item.getKey),
              item => item.getValue,
              item => item.isPassword
            )
          case LIST_OF_CI if pd.getName == "mappedProperties" =>
            diffListsOf[MappedProperty](
              oldValue.asInstanceOf[util.List[MappedProperty]].asScala.toSeq,
              newValue.asInstanceOf[util.List[MappedProperty]].asScala.toSeq,
              item => item.targetProperty,
              item => item.formattedProperty,
              item => item.formattedValue,
              item => item.isInstanceOf[PasswordValue]
            )
          case STRING if pd.isPassword => Seq(ChangedProperty(pd.getLabel, MASK, MASK))
          case _ => Seq(ChangedProperty(pd.getLabel, oldValue, newValue))
        }
      }
    }.toList
  }
}
