package com.xebialabs.xlrelease.triggers.service

import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.repository.{ConfigurationRepository, TriggerRepository}
import com.xebialabs.xlrelease.service.FolderOperationValidator
import com.xebialabs.xlrelease.validation.FolderOperationReferenceCategory._
import com.xebialabs.xlrelease.validation.FolderOperationValidationError
import grizzled.slf4j.Logging
import org.springframework.data.domain.PageRequest
import org.springframework.stereotype.Component

import java.util
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Component
class TriggerFolderOperationValidator(configurationRepository: ConfigurationRepository, triggerRepository: TriggerRepository)
  extends FolderOperationValidator with Logging {

  override def validateMoveOperation(folderId: String, newParentId: String, messages: util.List[FolderOperationValidationError]): Unit = {
    val invalidRefs: collection.Set[BaseConfiguration] = getNonInheritedFolderTriggerReferences(folderId, newParentId)

    if (invalidRefs.nonEmpty) {
      invalidRefs.map(invalidRef => {
        val errorMessage = FolderOperationValidationError(
          invalidRef.getId,
          invalidRef.getTitle,
          CONFIGURATION_CATEGORY,
          s"The folder or its subfolders contain configuration reference '${invalidRef.getTitle}' on triggers not inherited by the destination folder"
        )
        messages.add(errorMessage)
      })
    }
  }

  override def validateDeleteOperation(folderId: String, messages: util.List[FolderOperationValidationError]): Unit = {
    val all = PageRequest.of(0, Int.MaxValue)
    val triggers = triggerRepository.findByFolderId(folderId, nestedFolders = true, pageable = all).getContent.asScala.toSeq

    val enabledTriggers = triggers.filter(t => t.isEnabled)
    if (enabledTriggers.nonEmpty) {
      enabledTriggers.map(trigger => {
        val errorMessage = FolderOperationValidationError(
          trigger.getId,
          trigger.getTitle,
          TRIGGER_CATEGORY,
          s"The folder or its subfolders contain active trigger '${trigger.getTitle}', please disable it before continuing"
        )
        messages.add(errorMessage)
      })
    }
  }

  private def getNonInheritedFolderTriggerReferences(folderId: String, newfolderId: String) = {
    val nonInheritedConfIds = configurationRepository.findAllNonInheritedTriggerReferences(folderId)
    val res: mutable.Set[BaseConfiguration] = mutable.Set()

    nonInheritedConfIds.foreach { confId =>
      val c = configurationRepository.read(confId).asInstanceOf[BaseConfiguration]
      if (!(c.getFolderId == null || // is global OK
        newfolderId.startsWith(c.getFolderId) || // is inherited from parent in scope of destination OK
        c.getFolderId.startsWith(folderId))) { // will move along because in same folder or child folder OK
        res.add(c)
      }
    }
    res
  }

}
