package com.xebialabs.xlrelease.triggers.event_based.validators

import com.google.common.base.Strings.isNullOrEmpty
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.validation.{ExtendedValidationContext, ValidationContext, Validator}
import com.xebialabs.xlrelease.domain.Trigger
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.triggers.event_based.EventBasedTrigger
import com.xebialabs.xlrelease.webhooks.mapping.MappedProperty.{PropertyValue, StringValue, VariableValue}
import com.xebialabs.xlrelease.webhooks.mapping.PropertyAddress

import scala.jdk.CollectionConverters._
import scala.util.matching.Regex

class EventBasedTriggerValidator extends Validator[EventBasedTrigger] {

  override def validate(trigger: EventBasedTrigger, context: ValidationContext): Unit = {
    val extendedContext = context.asInstanceOf[ExtendedValidationContext]
    val templateId = trigger.getTemplateId.orNull
    val releaseTitle = trigger.mappedProperties.asScala.find(_.targetProperty == "releaseTitle").collect {
      case x: StringValue => x.value
      case x: PropertyValue => x.sourceProperty
      case x: VariableValue => x.variableKey
    }.orNull
    checkTemplatePresence()
    checkTemplateFolderAndTriggerFolderMatch()
    checkTriggerFolderAndSourceFolderMatch()
    checkReleaseTitlePresence()
    checkMalformedVariableNamesInMappings()
    checkInvalidMappedPropertyValues(trigger, extendedContext)
    checkFilterIsValid()

    def checkTemplatePresence(): Unit = {
      if (isNullOrEmpty(templateId)) {
        extendedContext.error(trigger, "templateId", s"Trigger does not have associated template")
      }
    }

    def checkTemplateFolderAndTriggerFolderMatch(): Unit = {
      if (!isNullOrEmpty(trigger.getFolderId) && !isNullOrEmpty(templateId) && Ids.findFolderId(templateId) != trigger.getFolderId) {
        extendedContext.error(trigger, "templateId", s"Trigger folder id does not match template folder id")
      }
    }

    def checkTriggerFolderAndSourceFolderMatch(): Unit = {
      if (trigger.eventSource != null) {
        val folderId: String = trigger.eventSource.getProperty("folderId")
        if (!isNullOrEmpty(trigger.getFolderId) && !isNullOrEmpty(folderId) && !trigger.getFolderId.contains(Ids.getName(folderId))) {
          extendedContext.error(trigger, "eventSource", s"Couldn't set eventSource that doesn't belong to the folder['${trigger.getFolderId}']")
        }
      }
    }

    def checkReleaseTitlePresence(): Unit = {
      if (isNullOrEmpty(releaseTitle)) {
        extendedContext.error(trigger, "releaseTitle", "Release title is required")
      }
    }

    def checkMalformedVariableNamesInMappings(): Unit = {
      if (trigger.isEnabled && containsMalformedVariableNames(trigger)) {
        extendedContext.focus(trigger, "mappedProperties").error(
          s"""There is an issue with the name of one or more variables.
             | Please update the variable names and save the trigger again.
             | The trigger will be disabled until the issue is fixed.""".stripMargin
        )
      }
    }

    def checkFilterIsValid(): Unit = {
      if (trigger.getEventFilter != null) {
        trigger.getEventFilter.validate(extendedContext)
      }
    }

  }

  private def containsMalformedVariableNames(trigger: Trigger): Boolean = {
    val badVarNamePatter: Regex = "variables\\[(.*(\\[|\\]).*)\\]".r
    trigger.asInstanceOf[EventBasedTrigger].mappedProperties.asScala.exists(mappedProp => {
      badVarNamePatter.findFirstMatchIn(mappedProp.targetProperty) match {
        case Some(_) => true
        case None => false
      }
    })
  }

  private def checkInvalidMappedPropertyValues(trigger: EventBasedTrigger, extendedContext: ExtendedValidationContext): Unit = {
    val invalidPropertyValues = trigger.mappedProperties.asScala.collect {
      case propertyValue: PropertyValue if !PropertyAddress.hasProperty(Type.valueOf(trigger.eventType), PropertyAddress(propertyValue.sourceProperty)) =>
        propertyValue
    }
    if (invalidPropertyValues.nonEmpty) {
      extendedContext.focus(trigger, "mappedProperties").error(
        s"""|Mapped properties contain references to non-existing event properties:
            |${invalidPropertyValues.map(pv => s"'${trigger.eventType}' has no '${pv.sourceProperty}'").mkString(",\n")}""".stripMargin
      )
    }
  }
}
