package com.xebialabs.xlrelease.triggers.actors

import com.xebialabs.deployit.plugin.api.validation.ValidationMessage
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.triggers.action.{TriggerAction, TriggerActionResult}
import com.xebialabs.xlrelease.triggers.service.impl.{TriggerExecutionContext, TriggerReferencedPermissionsValidator}
import com.xebialabs.xlrelease.validation.ExtendedValidationContextImpl
import grizzled.slf4j.Logging
import org.springframework.core.Ordered

import java.util.{List => JList}
import scala.util.{Failure, Success, Try}

case class TriggerExecutionResult[T <: Trigger](trigger: T, results: Try[TriggerActionResult])

// implementations of TriggerLifecyle:
//  - should be called from within TriggerOperations
trait TriggerLifecycle[T <: Trigger] extends Ordered with Logging {

  def supports(trigger: Trigger): Boolean

  def enable(trigger: T, checkReferencePermissions: Boolean): Unit

  def disable(trigger: T): Unit

  def validate(trigger: T, checkReferencePermissions: Boolean): JList[ValidationMessage] = {
    val extendedValidationContext = new ExtendedValidationContextImpl(trigger)
    trigger.getType.getDescriptor.validate(extendedValidationContext, trigger)

    if (checkReferencePermissions) {
      val permissionsValidator = new TriggerReferencedPermissionsValidator
      permissionsValidator.validate(trigger, extendedValidationContext)
    }
    extendedValidationContext.getMessages
  }

  def execute(trigger: T, triggerExecutionContext: TriggerExecutionContext): TriggerExecutionResult[T]

  def getTriggerAction(trigger: T, triggerExecutionContext: TriggerExecutionContext): TriggerAction

  final def runAction(trigger: T, executionContext: TriggerExecutionContext): Try[TriggerActionResult] = {
    val triggerId = trigger.getId
    Try {
      val action = getTriggerAction(trigger, executionContext)
      // TODO validate action (after all properties are mapped) before run
      runAction(action, executionContext) match {
        case Failure(ex) =>
          logger.warn(s"Unable to execute action of trigger $triggerId", ex)
          throw new RuntimeException(ex)
        case Success(actionResult) => actionResult
      }
    }
  }

  private def runAction(action: TriggerAction, executionContext: TriggerExecutionContext): Try[TriggerActionResult] = Try(action.run())

}
