package com.xebialabs.xlrelease.triggers.scheduled

import com.google.common.base.Strings.isNullOrEmpty
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.domain.{ReleaseTrigger, ScheduledTrigger, Trigger}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.{CiCloneHelper, Ids, ReleaseRepository, TriggerRepository}
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.service.ReleaseService
import com.xebialabs.xlrelease.triggers.action.{CreateReleaseFromTemplateAction, TriggerAction, TriggerActionSkipped}
import com.xebialabs.xlrelease.triggers.actors.TriggerActor.UpdateInternalState
import com.xebialabs.xlrelease.triggers.actors.{TriggerExecutionResult, TriggerLifecycle}
import com.xebialabs.xlrelease.triggers.config.TriggerExecutionConfiguration.TriggerActorHolder
import com.xebialabs.xlrelease.triggers.service.impl.TriggerExecutionContext
import com.xebialabs.xlrelease.validation.ExtendedValidationContextImpl
import grizzled.slf4j.Logging

import java.util
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

class ReleaseTriggerLifecycle(val scheduledJobService: ScheduledJobService,
                              val triggerScriptService: TriggerScriptService,
                              val releaseActorService: ReleaseActorService,
                              val releaseService: ReleaseService,
                              val eventBus: XLReleaseEventBus,
                              val triggerRepository: TriggerRepository,
                              val releaseRepository: ReleaseRepository,
                              val triggerActorHolder: TriggerActorHolder
                             )
  extends ScheduledTriggerLifecycle[ReleaseTrigger] {

  override def supports(trigger: Trigger): Boolean = trigger.isInstanceOf[ReleaseTrigger]

  override def getOrder: Int = 5

  override def getTriggerAction(trigger: ReleaseTrigger, triggerExecutionContext: TriggerExecutionContext): TriggerAction = {
    import com.xebialabs.xlrelease.variable.VariableHelper._

    // this code here is the equivalent of the mapping on event based triggers
    val title = replaceAll(trigger.getReleaseTitle, trigger.getStringScriptVariableValues)
    val tags = trigger.getTags.asScala.map(replaceAll(_, trigger.getStringScriptVariableValues)).filter(!containsVariables(_)).asJava
    val templateVariables = trigger.getTemplateVariableValues(!_.isPassword).asScala.collect { case (k, v) =>
      (k, replaceAll(v, trigger.getStringScriptVariableValues, new java.util.HashSet[String](), freezeEvenIfUnresolved = false))
    } ++ trigger.getTemplatePasswordVariables.asScala
    val clonedVariables = CiCloneHelper.cloneCis(trigger.getVariables)
    clonedVariables.forEach(v => templateVariables.get(v.getKey).foreach(v.setUntypedValue))

    val action = new CreateReleaseFromTemplateAction()
    action.setReleaseActorService(releaseActorService)
    action.setTriggerId(trigger.getId)
    action.setReleaseTitle(title)
    action.setTemplateId(trigger.getTemplate)
    action.setVariables(clonedVariables)
    action.setTags(tags)
    action.setReleaseFolderId(Option(trigger.getReleaseFolder).getOrElse(Ids.getParentId(trigger.getTemplate)))
    action
  }

  override def validate(trigger: ReleaseTrigger, checkReferencePermissions: Boolean = true): util.List[ValidationMessage] = {
    val msgs = super.validate(trigger, checkReferencePermissions)
    val extendedValidationContext = new ExtendedValidationContextImpl(trigger)
    val templateId = trigger.getTemplate
    val template = releaseService.findById(templateId, ResolveOptions.WITH_DECORATORS)

    def validateTemplateStatus(): Unit = {
      if (template.getStatus != ReleaseStatus.TEMPLATE) {
        extendedValidationContext.error(trigger, "template", "Cannot add a trigger that does not reference a TEMPLATE");
      }
    }

    def validateVariableValue(): Unit = {
      val variablesRequired = template.getVariables.asScala
        .filter(v => v.getRequiresValue && v.getShowOnReleaseStart && v.isValueEmpty)
        .map(v => v.getKey)
      val triggerVariablesFilled = trigger.getVariables.asScala
        .filter(v => !v.isValueEmpty)
        .map(v => v.getKey)
      val missingRequiredVariables = variablesRequired.diff(triggerVariablesFilled)
      if (missingRequiredVariables.nonEmpty) {
        extendedValidationContext.error(trigger, "variables",
          s"Missing value on trigger [${trigger.getTitle}] for required variables: [${missingRequiredVariables.mkString(",")}]")
      }
    }

    def validateVariableKeys(): Unit = {
      val templateVariables = template.getVariables.asScala.map(v => v.getKey)
      val triggerVariables = trigger.getVariables.asScala.map(v => v.getKey)
      val missingVariables = triggerVariables.diff(templateVariables)
      if (missingVariables.nonEmpty) {
        extendedValidationContext.error(trigger, "variables", s"Cannot find Variables: [${missingVariables.mkString(",")}] on template [${template.getTitle}]");
      }
    }

    def validateVariableTypes(): Unit = {
      val templateVariablesTypes = template.getVariables.asScala.map(v => v.getType)
      val triggerVariablesTypes = trigger.getVariables.asScala.map(v => v.getType)
      val missingVariablesTypes = triggerVariablesTypes.diff(templateVariablesTypes)
      if (missingVariablesTypes.nonEmpty) {
        extendedValidationContext.error(trigger, "variables", s" Cannot find Variables Type: " +
          s"[${missingVariablesTypes.mkString(",")}] on template [${template.getTitle}]")
      }
    }

    validateTemplateStatus()
    validateVariableValue()
    validateVariableKeys()
    validateVariableTypes()

    msgs.addAll(extendedValidationContext.getMessages)
    msgs
  }

  def evaluate(trigger: ReleaseTrigger, executionContext: TriggerExecutionContext): Either[String, ReleaseTrigger] = {
    logger.debug(s"Evaluating condition of trigger '${trigger.getId}'")
    // TODO for queries we do not have to go via actors
    // Doing this check here to avoid evaluating trigger script if releases can't be triggered
    val template = releaseService.findById(trigger.getTemplate, ResolveOptions.WITHOUT_DECORATORS)
    val blockerReason = releaseService.getTriggerReleaseBlocker(template)
    blockerReason match {
      case Some(reason) => Left(reason)
      case None => Right(triggerScriptService.executeTrigger(trigger))
    }
  }

}


trait ScheduledTriggerLifecycle[T <: ScheduledTrigger] extends TriggerLifecycle[T] with Logging {

  import com.xebialabs.xlrelease.triggers._

  def scheduledJobService: ScheduledJobService

  def triggerActorHolder: TriggerActorHolder

  override def supports(trigger: Trigger): Boolean = trigger.getType.isSubTypeOf(Type.valueOf(classOf[ScheduledTrigger]))

  override def getOrder: Int = 10

  override def enable(trigger: T, checkReferencePermissions: Boolean): Unit = scheduledJobService.schedule(trigger)

  override def disable(trigger: T): Unit = scheduledJobService.unschedule(trigger)

  def evaluate(t: T, executionContext: TriggerExecutionContext): Either[String, T]

  override def execute(trigger: T, executionContext: TriggerExecutionContext): TriggerExecutionResult[T] = {
    Try {
      evaluate(trigger, executionContext).map { updatedTrigger =>
        updateTriggerInternalState(updatedTrigger)
        updatedTrigger
      }
    } match {
      case Failure(error) =>
        logger.warn(s"Unable to poll trigger ${trigger.getId}", error)
        TriggerExecutionResult(trigger, Failure(error))
      case Success(Left(skipReason)) =>
        TriggerExecutionResult(trigger, Success(TriggerActionSkipped(skipReason)))
      case Success(Right(updatedTrigger)) =>
        val actionResult = if (hasTriggerStateChanged(trigger, updatedTrigger)) {
          logStateChange(trigger, updatedTrigger)
          trace(s"Running trigger action for: ${trigger.toJson}")
          runAction(updatedTrigger, executionContext)
        } else {
          Success(TriggerActionSkipped("Trigger state not changed"))
        }
        TriggerExecutionResult(updatedTrigger, actionResult)
    }
  }

  private def updateTriggerInternalState(updatedTrigger: T): Unit = {
    // this runs in a future created within an actor and uses cloned trigger value
    // to update actor we have to blockingly send a message to it
    // if trigger actor fails to update then this trigger execution also fails
    trace(s"Updating internal state of a trigger to ${updatedTrigger.toJson}")
    val varProperties = updatedTrigger.getType.getDescriptor.getPropertyDescriptors.asScala
      .filter(ReleaseTrigger.SCRIPT_VARS_CATEGORY == _.getCategory)
      .map(_.getName)
    val internalStateProperties: Seq[String] = Seq("triggerState") ++ varProperties
    triggerActorHolder.askAndAwait(UpdateInternalState(updatedTrigger, internalStateProperties))
  }

  private def hasTriggerStateChanged(previous: ScheduledTrigger, current: ScheduledTrigger): Boolean = {
    trace(s"Check if trigger state has changed. Previous: ${previous.toJson}. Current: ${current.toJson} ")
    (!isNullOrEmpty(previous.getTriggerState) || current.isInitialFire) && previous.getTriggerState != current.getTriggerState
  }

  private def logStateChange(previous: Trigger, current: Trigger): Unit = {
    val triggerId = previous.getId
    val triggerType = current.getType
    val currentState = current.getTriggerState
    val previousState = previous.getTriggerState
    debug(s"State changed on trigger : '$triggerId' of type '$triggerType'. New polling state: '$currentState', old state: '$previousState''")
  }
}
