package com.xebialabs.xlrelease.upgrade

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain
import com.xebialabs.xlrelease.domain.Trigger
import com.xebialabs.xlrelease.repository.sql.persistence.ReleasePersistence
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{TEMPLATE_TRIGGERS, TRIGGERS}
import com.xebialabs.xlrelease.repository.sql.{DeserializationSupport, SqlRepositoryAdapter}
import com.xebialabs.xlrelease.repository.{Ids, ReleaseRepository, TriggerRepository}
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.triggers.actors.TriggerLifecycle
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.UpgradeSupport.{BatchSupport, TransactionSupport}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.support.TransactionTemplate

import java.sql.ResultSet
import java.util
import scala.jdk.CollectionConverters._


private case class ReleaseTrigger(r: Int, t: Int)

@Component
class XLRelease960TriggerUpgrade @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                              @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                              @Qualifier("xlrRepositoryTransactionTemplate") val transactionTemplate: TransactionTemplate,
                                              val repositoryAdapter: SqlRepositoryAdapter,
                                              releasePersistence: ReleasePersistence,
                                              releaseRepository: ReleaseRepository,
                                              triggerRepository: TriggerRepository,
                                              ciIdService: CiIdService,
                                              triggerLifecycle: TriggerLifecycle[Trigger])
  extends Upgrade with Logging with BatchSupport with TransactionSupport with DeserializationSupport {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "9.6.0#1")


  override def doUpgrade(): Boolean = {


    val rows = jdbcTemplate.query(s"SELECT ${TEMPLATE_TRIGGERS.RELEASE_UID}, ${TEMPLATE_TRIGGERS.TRIGGER_UID} FROM ${TEMPLATE_TRIGGERS.TABLE}",
      new RowMapper[ReleaseTrigger] {
        override def mapRow(rs: ResultSet, rowNum: Int): ReleaseTrigger = ReleaseTrigger(rs.getInt(1), rs.getInt(2))
      }).asScala

    val groupedByRelease = rows.groupBy(f => f.r)
    doInBatch(groupedByRelease) {
      batch => {
        doInTransaction {
          batch.items.foreach {
            case (templateUid, _) =>
              val templateData = releasePersistence.findByReleaseUid(templateUid)
              templateData.map(templateRow => {
                logger.info(s"Upgrading release triggers from release [${templateRow.releaseId}]")
                val template = deserializeRelease(templateRow) // let's load all dependencies
                // this works as long as we have release triggers on Release level
                template.getReleaseTriggers.asScala.foreach(trigger => {
                  // set the folder from the template
                  trigger.setFolderId(templateRow.folderId)
                  // set template for trigger
                  trigger.setTemplate(template.getId)
                  // replace Release13/Trigger123 with Trigger123
                  val oldId = Ids.getFolderlessId(trigger.getId)
                  // generate a new trigger Id as trigger Ids can be same if template is copied.
                  // For example, Release123/Trigger123 is copied and a new Id for this will be Release456/Trigger123
                  val newId = Ids.getFolderlessId(ciIdService.getUniqueId(Type.valueOf(classOf[Trigger]), ""))
                  // we need to get the CiUID from database as template.getReleaseTriggers doesn't have
                  val ciUid = triggerRepository.find[Trigger](oldId).getCiUid
                  jdbcTemplate.update(s"UPDATE ${TRIGGERS.TABLE} SET ${TRIGGERS.ID} = '$newId' WHERE ${TRIGGERS.ID} = '$oldId'")
                  // update content json
                  trigger.setId(newId)
                  // used to create link on TriggerPersistenceInterceptor
                  trigger.setCiUid(ciUid)
                  val updatedTrigger = triggerRepository.update(trigger)
                  //Call trigger lifecycle if trigger is enabled
                  if (trigger.isEnabled) {
                    triggerLifecycle.enable(updatedTrigger, true)
                  }
                })
                // remove the release triggers from the release json
                template.setReleaseTriggers(new util.ArrayList[domain.ReleaseTrigger]())
                releaseRepository.update(template)
              })
            case _ => logger.error("oh snap! :(")
          }
        }
      }
    }
    true
  }


}
