package com.xebialabs.xlrelease.webhooks.registry

import com.xebialabs.xlplatform.webhooks.events.domain.EventConsumer.CiEventConsumer
import com.xebialabs.xlplatform.webhooks.events.domain.{EventConsumer, EventSource}
import com.xebialabs.xlrelease.webhooks.consumers.logging.SpyConsumer
import org.slf4j.LoggerFactory

import scala.collection.mutable

abstract class SubscriberRegistry {
  private val logger = LoggerFactory.getLogger(classOf[SubscriberRegistry])

  private val subscribers: mutable.Map[EventSource.ID, mutable.Set[CiEventConsumer]] = mutable.Map.empty

  def subscribe(sourceId: EventSource.ID, consumerConfig: CiEventConsumer): Unit = {
    logger.info(s"Subscribing consumer `$consumerConfig` to source `$sourceId`")
    if (consumerConfig.consumerEnabled) {
      subscribers.synchronized {
        subscribers.getOrElse(sourceId, {
          val newSet = mutable.Set.empty[CiEventConsumer]
          subscribers += sourceId -> newSet
          newSet
        }) += consumerConfig
      }
    }
  }

  def unsubscribe(sourceId: EventSource.ID, consumerId: EventConsumer.ID): Unit = {
    subscribers.synchronized {
      subscribers.get(sourceId).foreach(_.filterInPlace(_.consumerId != consumerId))
    }
    logger.info(s"Unsubscribed consumer `$consumerId` from source `$sourceId`")
  }

  def remove(consumerId: EventConsumer.ID): Unit = {
    subscribers.synchronized {
      subscribers.values.foreach(_.filterInPlace(_.consumerId != consumerId))
    }
    logger.info(s"Removed consumer `$consumerId` from all sources")
  }

  def list(sourceId: EventSource.ID): Set[CiEventConsumer] = {
    subscribers.synchronized {
      (subscribers.getOrElse(sourceId, mutable.Set.empty) + SpyConsumer()).toSet
    }
  }

  def listAll: Map[EventSource.ID, Set[CiEventConsumer]] = {
    subscribers.synchronized {
      subscribers.view.mapValues(_.toSet).toMap
    }

  }
}
