package com.xebialabs.xlrelease.webhooks.configuration.jms

import com.typesafe.config.Config
import com.xebialabs.xlrelease.server.jetty.SSLConstants
import org.slf4j.{Logger, LoggerFactory}

import java.io.FileInputStream
import java.security.{KeyStore, SecureRandom}
import javax.net.ssl.{KeyManagerFactory, SSLContext, TrustManagerFactory}
import scala.util.Using

case class SslContextConfig(
                           debug: Boolean,
                           protocol: String, // TLSv1.2
                           trustEverything: Boolean,
                           hostnameVerification: Boolean,
                           keystoreType: String, // jks
                           keystorePath: String,
                           keystorePass: String,
                           truststoreType: String, // jks
                           truststorePath: String,
                           truststorePass: String
                           )

object SslContextConfig {
  import com.xebialabs.xlrelease.support.config.TypesafeConfigExt._

  def fromConfig(config: Config): SslContextConfig = {
    val debug = config.getOptionalBoolean("debug").getOrElse(false)
    val protocol = config.getOptionalString("protocol").getOrElse("TLSv1.2")
    val trustEverything = config.getOptionalBoolean("trustEverything").getOrElse(true)
    val hostnameVerification = config.getOptionalBoolean("hostnameVerification").getOrElse(false)
    val keystoreType = config.getOptionalString("keyStoreType").getOrElse(KeyStore.getDefaultType)
    val keystorePath = config.getOptionalString("keyStore").getOrElse(System.getProperty(SSLConstants.KEYSTORE_PROPERTY))
    val keystorePass = config.getOptionalString("keyStorePassword").getOrElse(System.getProperty(SSLConstants.KEYSTORE_PASSWORD_PROPERTY))
    val truststorePath = config.getOptionalString("trustStore").getOrElse(System.getProperty(SSLConstants.TRUSTSTORE_PROPERTY))
    val truststorePass = config.getOptionalString("trustStorePassword").getOrElse(System.getProperty(SSLConstants.TRUSTSTORE_PASSWORD_PROPERTY))
    val truststoreType = config.getOptionalString("trustStoreType").getOrElse(KeyStore.getDefaultType)
    SslContextConfig(
      debug = debug,
      protocol = protocol,
      trustEverything = trustEverything,
      hostnameVerification = hostnameVerification,
      keystoreType = keystoreType,
      keystorePath = keystorePath,
      keystorePass = keystorePass,
      truststoreType = truststoreType,
      truststorePath = truststorePath,
      truststorePass = truststorePass
    )
  }
}

trait SslContextFactory {
  def logger: Logger = LoggerFactory.getLogger(classOf[SslContextFactory])
  def getContext(config: SslContextConfig): SSLContext
}

object SslContextFactory extends SslContextFactory {

  override def getContext(config: SslContextConfig): SSLContext = {
    sslContext(config)
  }

  private def sslContext(config: SslContextConfig): SSLContext = {
    if (config.debug) {
      System.setProperty("javax.net.debug", "ssl")
    }
    try {
      // keystore
      val keyManagers = if (null != config.keystorePath) {
        val keystorePassphrase: Array[Char] = config.keystorePass.toCharArray
        val ks = KeyStore.getInstance(config.keystoreType);
        Using(new FileInputStream(config.keystorePath)) { in =>
          ks.load(in, keystorePassphrase)
        }
        val kmf: KeyManagerFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm)
        kmf.init(ks, keystorePassphrase)
        kmf.getKeyManagers
      } else {
        null
      }
      // truststore
      val trustManagers = if (config.truststorePath != null) {
        val tks: KeyStore = KeyStore.getInstance(config.keystoreType)
        Using(new FileInputStream(config.truststorePath)) { in =>
          val truststorePassphrase: Array[Char] = config.truststorePass.toCharArray
          tks.load(in, truststorePassphrase)
        }
        val tmf: TrustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm)
        tmf.init(tks)
        tmf.getTrustManagers
      } else {
        null
      }
      // https://docs.oracle.com/en/java/javase/21/docs/specs/security/standard-names.html#securerandom-number-generation-algorithms
      // by default, it is going to use new SecureRandom()
      val context = if (trustManagers != null) {
        val ctxt =  SSLContext.getInstance(config.protocol)
        val secureRandom = new SecureRandom()
        ctxt.init(keyManagers, trustManagers, secureRandom)
        ctxt
      } else {
        // just use default ssl context
        SSLContext.getDefault
      }
      context
    } catch {
      case e: Throwable =>
        val msg = "Unable to create SSL context"
        logger.error(msg, e)
        throw new IllegalStateException(msg, e)
    }
  }

}
