package com.xebialabs.xlrelease.webhooks.task.script

import com.xebialabs.deployit.security.Permissions.getAuthentication
import com.xebialabs.platform.script.jython.ScriptSource
import com.xebialabs.xlplatform.webhooks.events.domain.Event
import com.xebialabs.xlrelease.domain.{CustomScriptTask, PythonScript}
import com.xebialabs.xlrelease.script._
import com.xebialabs.xlrelease.script.jython.JythonScriptService
import com.xebialabs.xlrelease.webhooks.task._
import com.xebialabs.xlrelease.webhooks.untyped.EventContentParser
import grizzled.slf4j.Logging
import org.apache.commons.lang3.StringUtils.isNotBlank
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.util.Using

@Component
class ReactiveEventFilter @Autowired()(eventContentParser: EventContentParser,
                                       scriptVariables: ScriptVariables,
                                       scriptService: JythonScriptService) extends Logging {

  def matches(task: CustomScriptTask, event: Event): ReactiveTaskResult = {
    val expr = getScript(task.getPythonScript).scriptContent
    Option(expr) match {
      case Some(expr) if isNotBlank(expr) =>
        Using.resource(createContext(task, event, expr)) { context =>
          scriptService.executeGenericTaskScript(task, context)
        }
      case _ =>
        FailedEventResult(task.getId, task.getExecutionId, "No condition script defined", None, getAuthentication, None)
    }
  }

  private def createContext(task: CustomScriptTask, event: Event, expr: String): XlrScriptContext = {
    val release = task.getRelease
    val folderId = release.findFolderId()
    val variablesHolder = scriptVariables.createVariablesHolderForScriptContext(release, folderId)
    val vars = scriptVariables.asXlrScriptVariables(variablesHolder)
    val builder = new ReactiveEventFilterScriptContextBuilder(task, event, eventContentParser, expr)
    builder.withGlobalVariables(vars.globalVariables)
    builder.withFolderVariables(vars.folderVariables)
    builder.withReleaseVariables(vars.releaseVariables)
    builder.build()
  }

  private def getScript(pythonScript: PythonScript): ScriptSource = {
    if (pythonScript.hasProperty(SCRIPT_PROPERTY)) {
      ScriptSource.byContent(pythonScript.getProperty(SCRIPT_PROPERTY))
    } else if (pythonScript.hasProperty(SCRIPT_LOCATION_PROPERTY)) {
      ScriptSource.byResource(pythonScript.getProperty(SCRIPT_LOCATION_PROPERTY))
    } else {
      throw new RuntimeException(s"Neither $SCRIPT_PROPERTY nor $SCRIPT_LOCATION_PROPERTY are defined")
    }
  }
}
