package com.xebialabs.xlrelease.webhooks.mapping

case class PropertyDef private(name: String, key: Option[String]) {
  override def toString: String =
    name + key.fold("")("[" + _ + "]")
}

object PropertyDef {
  def apply(address: String): PropertyDef = address match {
    case null =>
      throw new NullPointerException("PropertyDef address cannot be null")

    case elem if !isValidPart(elem) =>
      throw new IllegalArgumentException(s"Single property definition should be in format 'property' or 'property[key]'")

    case elem if elem.contains('[') && elem.contains(']') =>
      val l = elem.split('[')
      require(l.length == 2)
      require(l(1).last == ']')
      checkValidName(l.head)
      new PropertyDef(l.head, Some(l(1).init))

    case elem =>
      checkValidName(elem)
      new PropertyDef(elem, None)
  }

  def isValidPart(part: String): Boolean = Option(part)
    .filter(s =>
      // no index subscript
      (s.count(_ == '[') == 0 && s.count(_ == ']') == 0) ||
      (s.count(_ == '[') == 1 && s.count(_ == ']') == 1 && {
        val start = s.indexOf('[')
        val end = s.indexOf(']')
        end == s.length - 1 && // must end with ']'
          start > 0 &&
          end >= start + 1 // cannot start '['. ']any[' is invalid
      })
    )
    .exists(_.nonEmpty)

  def checkValidName(name: String): Unit = if (!isValidName(name)) {
    throw new IllegalArgumentException(s"Property name contains illegal characters")
  }

  def isValidName(name: String): Boolean = Option(name)
    .exists(_.forall(validChars.contains))

  // TODO: make sure this is aligned with what platform does.
  lazy val validChars: Set[Char] =
    ('a' to 'z').toSet union
      ('A' to 'Z').toSet union
      ('0' to '9').toSet union
      Set('_', '[', ']', ' ')
}
