# Copyright 2024 XEBIALABS
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

import com.xhaus.jyson.JysonCodec as json
from xlrelease.HttpRequest import HttpRequest
from airflow.util import error
from airflow.util import validate_mandatory_field
from urllib import urlencode

class AirflowServer(object):
    __json = 'application/json'

    def __init__(self, airflowServer):
        validate_mandatory_field(airflowServer, "Airflow Server not configured")
        self.request = HttpRequest(airflowServer)

    def getDag(self, dagId):
        validate_mandatory_field(dagId, "dagID is not configured")
        response = self.request.get('/api/v1/dags/%s' % (dagId), contentType=self.__json)

        if response.status != 200:
            error("Unexpected error happened while fetching DAG details", response)

        response_body = json.loads(response.response)
        return response_body    
    
    def updateDag(self, dagId, isActive):
        validate_mandatory_field(dagId, "dagID is not configured")
        updateUrl = '/api/v1/dags/%s' % dagId
        payload = {
            "is_paused" : not isActive
        }
        response = self.request.patch(updateUrl, json.dumps(payload), contentType=self.__json)
        if response.status != 200:
            error("Unexpected error happened while updating the DAG", response)
        response_body = json.loads(response.response)
        return response_body  

    def getDagRun(self, dagId, dagRunId):
        validate_mandatory_field(dagId, "dagID is not configured")
        validate_mandatory_field(dagRunId, "dagRunID is not configured")

        response = self.request.get('/api/v1/dags/%s/dagRuns/%s' % (dagId, dagRunId), contentType=self.__json)

        if response.status != 200:
            error("Unexpected error happened while fetching DAG run", response)

        response_body = json.loads(response.response)
        return response_body

    def getTaskInstance(self, dagId, dagRunId, taskId):
        validate_mandatory_field(dagId, "dagID is not configured")
        validate_mandatory_field(dagRunId, "dagRunID is not configured")
        validate_mandatory_field(taskId, "taskID is not configured")

        response = self.request.get('/api/v1/dags/%s/dagRuns/%s/taskInstances/%s' % (dagId, dagRunId, taskId), contentType=self.__json)

        if response.status != 200:
            error("Unexpected error happened while fetching task instance", response)

        response_body = json.loads(response.response)
        return response_body

    def triggerDagRun(self, dagId, dagRunId):
        validate_mandatory_field(dagId, "dagID is not configured")

        payload = {}
        if dagRunId:
            payload['dag_run_id'] = dagRunId
        response = self.request.post('/api/v1/dags/%s/dagRuns' % dagId, json.dumps(payload), contentType=self.__json)

        if response.status != 200:
            error("Unexpected error happened while triggering DAG run", response)

        response_body = json.loads(response.response)
        return response_body

    def listDags(self, dagId):
        response = self.request.get('/api/v1/dags?dag_id_pattern=%s' % dagId, contentType=self.__json)
        if response.status != 200:
            return []
        response_body = json.loads(response.response)
        return response_body
    
    def filterDags(self, dag_id_pattern, tags):
        baseUrl = '/api/v1/dags'
        queryParams = {}
        if dag_id_pattern:
           queryParams["dag_id_pattern"] = dag_id_pattern
        if tags:
            queryParams["tags"] = tags
        requestUrl = baseUrl + "?" + urlencode(queryParams, doseq=True)
        response = self.request.get(requestUrl, contentType=self.__json)
       
        if response.status != 200:
            error("Unexpected error happened while fetching DAG's", response)
        response = json.loads(response.response)
        dagMap = {}
        for dag in response['dags']:
            dag_id = dag['dag_id']
            description = dag['description']
            dagMap[dag_id] = description if description else " "
        return dagMap

    def listTasks(self, dagId):
        response = self.request.get('/api/v1/dags/%s/tasks' % dagId, contentType=self.__json)
        if response.status != 200:
            error("Unexpected error happened while fetching tasks", response)
        response_body = json.loads(response.response)
        return response_body

    def listDagRunIds(self, dagId):
        validate_mandatory_field(dagId, "dagID is not configured")
        response = self.request.get('/api/v1/dags/%s/dagRuns' % dagId, contentType=self.__json)
        if response.status != 200:
            error("Unexpected error happened while fetching DAG run IDs", response)
        response_body = json.loads(response.response)
        return response_body
    
    def modifyDagRun(self, dagId, dagRunId, state):
        validate_mandatory_field(dagId, "dagID is not configured")
        validate_mandatory_field(dagRunId, "dagRunID is not configured")

        payload = {'state': state}
        response = self.request.patch('/api/v1/dags/%s/dagRuns/%s' % (dagId, dagRunId), json.dumps(payload),contentType=self.__json)

        if response.status != 200:
            error("Unexpected error happened while modifying DAG run", response)
        response_body = json.loads(response.response)
        return response_body
    
    def deleteDagRun(self, dagId, dagRunId):
        validate_mandatory_field(dagId, "dagID is not configured")
        validate_mandatory_field(dagRunId, "dagRunID is not configured")

        response = self.request.delete('/api/v1/dags/%s/dagRuns/%s' % (dagId, dagRunId))
        
        if response.status != 204:
            error("Unexpected error happened while deleting DAG run", response)