#
# Copyright (c) 2021. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of Digital.ai, Inc. and its affiliates, subsidiaries, and licensors.
#
import json
from xlrelease.HttpRequest import HttpRequest

from com.xebialabs.xlrelease.plugin.argocd import LoaderUtil


class RestClient:

    def __init__(self, argo_server=None):
        self.httpRequest  = HttpRequest(argo_server, verify=False)
        self.username = argo_server['username']
        self.password = argo_server['password']
        self.auth_token = argo_server['authToken']

    def getBearerToken(self):
        if self.auth_token is not None and self.auth_token != "":
            print "The provided authToken can be used as a bearer token"
            return self.auth_token
        # Create POST body
        content = {
            'username': self.username,
            'password': self.password
        }
        response = self.httpRequest.post('/api/v1/session', json.dumps(content),
                                contentType='application/json', headers=None)
        if response.isSuccessful():
            response_json = json.loads(response.getResponse())
            bearer_token = response_json['token']
        else:
            raise ValueError('Failed to get bearer token %s:%s' % (response.getStatus(),response.getResponse()))
        return bearer_token

    def checkConnection(self):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        response = self.httpRequest.get('/api/v1/projects', contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to connect the server %s:%s' % (response.getStatus(), response.getResponse()))

    def listClusters(self):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        response = self.httpRequest.get('/api/v1/clusters', contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to list clusters %s:%s' % (response.getStatus(), response.getResponse()))

        responseObj = json.loads(response.getResponse())
        clusters = responseObj["items"]
        return clusters

    def listProjects(self):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        response = self.httpRequest.get('/api/v1/projects', contentType='application/json', headers=headers)
        result = {}
        projects = []
        if response.isSuccessful():
            responseObj = json.loads(response.getResponse())
            items = responseObj.get("items") if responseObj.get("items") is not None else []
            for item in items:
                project = {}
                spec = item["spec"]
                metadata = item["metadata"]
                project["name"] = metadata.get("name")
                project["namespace"] = metadata.get("namespace")
                project["description"] = spec.get("description")
                project["sourceRepos"] = spec.get("sourceRepos")
                project["destinations"] = spec.get("destinations")
                projects.append(project)
        else:
            raise ValueError('Failed to get project list %s:%s' % (response.getStatus(),response.getResponse()))
        result["projects"] = projects
        return result

    def listRepositories(self):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        response = self.httpRequest.get('/api/v1/repositories', contentType='application/json', headers=headers)
        repos = None
        if response.isSuccessful():
            repositories = json.loads(response.getResponse())
            repos = []
            items = repositories.get("items") if repositories.get("items") is not None else []
            for item in items:
                repo = item.get("repo")
                if repo is not None :
                    repos.append(repo)
        else:
            raise ValueError('Failed to get repositories list %s:%s' % (response.getStatus(),response.getResponse()))
        return repos

    def listApplications(self, projects):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        params = ''
        for project in projects:
            params += '&projects=' + project + '&project=' + project
        response = self.httpRequest.get('/api/v1/applications?' + params, contentType='application/json', headers=headers)
        applications = []
        if response.isSuccessful():
            responseObj = json.loads(response.getResponse())
            items = responseObj.get("items") if responseObj.get("items") is not None else []
            for item in items:
                metadata = item.get("metadata", {})
                if metadata.get("deletionTimestamp") is None:
                    spec = item.get("spec", {})
                    status = item.get("status", {})
                    application = {}
                    application["name"] = metadata.get("name")
                    application["path"] = "/applications/" + metadata.get("name") + "?view=tree"
                    application["created_at"] = metadata.get("creationTimestamp")
                    application["namespace"] = metadata.get("namespace", {})
                    application["labels"] = metadata.get("labels", {})
                    application["destination"] = spec.get("destination", {})
                    application["status"] = status

                    repo_url = spec.get("source", {}).get("repoURL")
                    application["repo_url"] = repo_url

                    revision = status.get("sync", {}).get("revision")
                    application["revision"] = revision
                    revision_metadata = self.get_revision_metadata(application["name"], revision) if revision is not None else {}
                    application["revision_metadata"] = revision_metadata if revision_metadata is not None else None

                    last_change = status.get("operationState", {}).get("finishedAt")
                    application["last_change"] = last_change if last_change is not None else metadata.get("creationTimestamp")

                    applications.append(application)
        else:
            raise ValueError('Failed to get application list %s:%s' % (response.getStatus(), response.getResponse()))
        return applications

    def addRepository(self, url, username, password, sshKey):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        if sshKey :
            body = {
                "url": url,
                "sshPrivateKey": sshKey
            }
        else:
            body = {
                "type": "git",
                "repo": url,
                "username": username,
                "password": password
            }
        response = self.httpRequest.post('/api/v1/repositories', json.dumps(body), contentType='application/json', headers=headers)

        if not response.isSuccessful():
           raise ValueError('Failed to add repository list %s:%s' % (response.getStatus(),response.getResponse()))

    def createApplication(self, appName, project, url, path, revision, server, namespace,
                          autoSync, pruneResources, selfHeal, skipValidation, pruneLast,
                          autoCreateNamespace, applyOutOfSyncOnly, prunePropagationPolicy,
                          replace, validate=False, upsert=False):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        body = {
            "metadata": {
                "name": appName
            },
            "spec": {
                "destination": {
                    "namespace": namespace,
                    "server": server
                },
                "project": project,
                "source": {
                    "path": path,
                    "repoURL": url,
                    "targetRevision": revision
                }
            }
        }
        syncPolicy = {}
        if autoSync:
            automated = {}
            automated["prune"] = pruneResources
            automated["selfHeal"] = selfHeal
            syncPolicy["automated"] = automated

        syncOptions = []
        syncOptions.append("Validate=%s" % (not skipValidation))
        syncOptions.append("PruneLast=%s" % (pruneLast))
        syncOptions.append("CreateNamespace=%s" % (autoCreateNamespace))
        syncOptions.append("ApplyOutOfSyncOnly=%s" % (applyOutOfSyncOnly))
        syncOptions.append("Replace=%s" % (replace))
        syncOptions.append("PrunePropagationPolicy=%s" % (prunePropagationPolicy))
        syncPolicy["syncOptions"] = syncOptions

        specObj = body["spec"]
        specObj["syncPolicy"] = syncPolicy
        body["spec"] = specObj

        url = "/api/v1/applications?validate=%s&upsert=%s" % (validate, upsert)
        print(url)
        response = self.httpRequest.post(url, json.dumps(body), contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to create app %s:%s' % (response.getStatus(),response.getResponse()))

        return response.getResponse()

    def syncApplication(self, appName, autoCreateNamespace):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}

        items = []
        if autoCreateNamespace:
            items.append("CreateNamespace=true")
        sync_options = {'items': items}
        body = {'syncOptions': sync_options}

        endpoint = "/api/v1/applications/" + appName + "/sync"
        response = self.httpRequest.post(endpoint, json.dumps(body), contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to sync app list %s:%s' % (response.getStatus(),response.getResponse()))
        else:
            responseObj = json.loads(response.getResponse())
            statusObj = responseObj["status"]
            syncObj = statusObj["sync"]
            status = syncObj["status"]
            revision = syncObj['revision']
            result = {
                "status" : status,
                "revision": revision,
            }
        return result

    def getSyncStatus(self, appName):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/" + appName
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)
        if not response.isSuccessful():
            raise ValueError('Failed to get sync status %s:%s' % (response.getStatus(),response.getResponse()))

        deployment_status = {}

        responseObj = json.loads(response.getResponse())
        statusObj = responseObj["status"]
        sync = statusObj["sync"]
        revision = sync["revision"]
        syncStatus = sync["status"]
        healthObj = statusObj["health"]
        healthStatus = healthObj["status"]
        syncFailed = False
        message = ""
        if "operationState" in statusObj :
            opState = statusObj["operationState"]
            if opState["phase"] == "Failed":
                syncFailed = True
                message = opState["message"]
        deployment_status = {
            "syncStatus": syncStatus,
            "healthStatus": healthStatus
        }
        result = {
            "status" : deployment_status,
            "revision" : revision,
            "syncFailed" : syncFailed,
            "message" : message
        }

        return result

    def createProject(self, projectName, sourceRepos, destinations):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        body = { "upsert": True }
        project = {
            "metadata": {
                "name": projectName
            }
        }
        spec = {}
        if sourceRepos:
            sourceRepoList = []
            for repo in sourceRepos:
                sourceRepoList.append(repo)
            spec["sourceRepos"] = sourceRepoList

        if destinations:
            destinationList = []
            for serverKey in destinations:
                destObj =  {
                    "namespace": destinations[serverKey],
                    "server": serverKey
                }
                destinationList.append(destObj)
            spec["destinations"] = destinationList

        clusterResourceWhitelist = [{
            "group": "*",
            "kind": "*"}]

        spec["clusterResourceWhitelist"] = clusterResourceWhitelist

        project["spec"] = spec
        body["project"] = project

        response = self.httpRequest.post('/api/v1/projects', json.dumps(body), contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to create project %s:%s' % (response.getStatus(),response.getResponse()))

        return response.getResponse()

    def getAppDetails(self, appName, bearer_token):
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/" + appName
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to get application details %s:%s' % (response.getStatus(),response.getResponse()))

        return json.loads(response.getResponse())

    def getResourceDetails(self, appName, resourceName, bearer_token):
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/" + appName + "/managed-resources?name="+resourceName
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to get resource details %s:%s' % (response.getStatus(),response.getResponse()))

        return json.loads(response.getResponse())

    def getResourceTree(self, appName):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/" + appName+"/resource-tree"
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to get sync status %s:%s' % (response.getStatus(), response.getResponse()))

        return json.loads(response.getResponse())

    def get_revision_metadata(self, appName, revision):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/%s/revisions/%s/metadata" % (appName, revision)
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)
        if not response.isSuccessful():
            raise ValueError('Failed to get revision metadata %s:%s' % (response.getStatus(), response.getResponse()))

        return json.loads(response.getResponse())

    def add_notification_annotations(self, projects):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        params = ''
        for project in projects:
            params += '&projects=' + project + '&project=' + project
        response = self.httpRequest.get('/api/v1/applications?' + params, contentType='application/json', headers=headers)

        if response.isSuccessful():
            response_obj = json.loads(response.getResponse())
            items = response_obj.get("items") if response_obj.get("items") is not None else []
            for item in items:
                app_name = item.get("metadata", {}).get("name")
                item['metadata']['annotations'] = item.get("metadata", {}).get('annotations', {})
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-created.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-deleted.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-health-unknown.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-health-progressing.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-health-suspended.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-health-healthy.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-health-degraded.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-health-missing.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-sync-unknown.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-sync-synced.release-webhook'] = ''
                item['metadata']['annotations']['notifications.argoproj.io/subscribe.on-sync-out-of-sync.release-webhook'] = ''

                update_annotations_response = self.httpRequest.put('/api/v1/applications/' + app_name, json.dumps(item), contentType='application/json', headers=headers)

                if not update_annotations_response.isSuccessful():
                    raise ValueError('Failed to update application %s:%s' % (response.getStatus(),response.getResponse()))

            return items
        else:
            raise ValueError('Failed to get application list %s:%s' % (response.getStatus(), response.getResponse()))

    @staticmethod
    def generateYaml(url, token):
        res = LoaderUtil.getResourceBySelfClassLoader("argocd/templates/configmap.yaml")
        file_data = ""

        if res:
            with open(res.openStream()) as file :
               for line in file:
                   if token == "" and (line.find("Authorization") != -1 or line.find("Bearer $TOKEN") != -1):
                       pass
                   else:
                       file_data += line

        file_data = file_data.replace("$XL_RELEASE_URL", url)
        file_data = file_data.replace("$TOKEN", token)

        return file_data
