package com.xebialabs.xlrelease.auth.oidc.policy.impl;

import java.util.Map;
import org.slf4j.Logger;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.Authentication;
import org.springframework.security.oauth2.core.oidc.user.OidcUser;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.security.UserService;
import com.xebialabs.xlrelease.domain.UserProfile;
import com.xebialabs.xlrelease.repository.CiCloneHelper;
import com.xebialabs.xlrelease.security.authentication.policy.UserProfileCreationPolicy;
import com.xebialabs.xlrelease.service.UserProfileService;

import static com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * Fill UserProfile from Open ID Connect Claims
 * <p>
 * http://openid.net/specs/openid-connect-core-1_0.html#StandardClaims
 */
public class OidcUserProfileCreationPolicy implements UserProfileCreationPolicy {
    private static final Logger logger = getLogger(OidcUserProfileCreationPolicy.class);

    private final String emailClaim;
    private final String fullNameClaim;
    private final String externalIdClaim;
    private final UserProfileService userProfileService;
    private final UserService userService;

    public OidcUserProfileCreationPolicy(
            UserProfileService userProfileService,
            UserService userService,
            String emailClaim,
            String fullNameClaim,
            String externalIdClaim
    ) {
        this.userProfileService = userProfileService;
        this.userService = userService;
        this.emailClaim = emailClaim;
        this.fullNameClaim = fullNameClaim;
        this.externalIdClaim = externalIdClaim;
    }

    @Override
    public boolean policyApplies(final Authentication authentication) {
        return authentication != null && authentication.getPrincipal() != null && authentication.getPrincipal() instanceof OidcUser;
    }

    @Override
    public int order() {
        return 10;
    }

    @Override
    public UserProfile createProfile(Authentication authentication) {
        String userName = authentication.getName();
        Map<String, Object> claims = ((OidcUser) authentication.getPrincipal()).getClaims();
        try {
            // Refuse to create an External User if a Local User exists
            userService.read(userName);
            logger.warn("Unable to create profile for user [{}]. A local account exists.", userName);
            throw new BadCredentialsException(String.format("A local account exists for user %s", userName));
        } catch (NotFoundException e) {
            UserProfile profile = userProfileService.findByUsername(userName);

            if (profile != null) {
                UserProfile original = CiCloneHelper.cloneCi(profile);
                setUserProfileProperties(profile, claims);
                if (UserProfileService.hasExternalPropertiesChanged(original, profile)) {
                    userProfileService.updateProfile(profile);
                }
            } else {
                profile = new UserProfile(userName);
                profile.setLoginAllowed(true);
                setUserProfileProperties(profile, claims);
                userProfileService.save(profile);
            }
            return profile;
        }
    }

    protected String getString(String field, Map<String, Object> data) {
        String v = (String) data.get(field);
        if (v == null) {
            logger.warn("Did not receive {} field from OIDC provider", field);
            return "";
        }
        return v;
    }

    private void setUserProfileProperties(UserProfile userProfile, Map<String, Object> claims) {
        String email = getString(emailClaim, claims);
        String fullName = getString(fullNameClaim, claims);

        if (isNotBlank(email)) {
            userProfile.setEmail(email);
        }
        if (isNotBlank(fullName)) {
            userProfile.setFullName(fullName);
        }
        if (isNotBlank(externalIdClaim)) {
            userProfile.setExternalId(getString(externalIdClaim, claims));
        }
    }
}
