package com.xebialabs.xlrelease.auth.oidc.web;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;

import com.xebialabs.deployit.util.MustacheRenderer;
import com.xebialabs.xlrelease.configuration.ThemeSettings;
import com.xebialabs.xlrelease.service.ConfigurationService;

public class XlReleaseLoginFormFilter extends UsernamePasswordAuthenticationFilter {

    private static final String POST = "POST";
    private static final String GET = "GET";
    private static final String BLACKLIST_RESOURCE = "favicon.ico";

    public static final String ERROR_PARAMETER_NAME = "error";
    public static final String LOGIN_PATH_NAME = "login";
    public static final String CONTENT = "content";
    public static final String PRODUCT_COLOR = "productColor";
    public static final String INSTANCE_NAME = "instanceName";

    private final RequestMatcher requiresAuthenticationRequestMatcher;
    private final Map<String, String> settings;
    private final Resource layout;
    private final Resource resourceContent;

    @Autowired
    private ConfigurationService configurationService;

    public XlReleaseLoginFormFilter(Map<String, String> settings) {
        this.requiresAuthenticationRequestMatcher = new AntPathRequestMatcher("/" + LOGIN_PATH_NAME);
        this.layout = new ClassPathResource("com/xebialabs/platform/sso/oidc/web/modal-layout.mustache");
        this.resourceContent = new ClassPathResource("com/xebialabs/xlrelease/auth/oidc/web/login.mustache");
        this.settings = settings;
    }

    @Override
    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain) throws AuthenticationServiceException, IOException, ServletException {
        final HttpServletRequest request = (HttpServletRequest) req;
        final HttpServletResponse response = (HttpServletResponse) res;

        if (requiresAuthenticationRequestMatcher.matches(request)) {
            if (request.getMethod().equals(GET)) {
                Map<String, Resource> partials = new HashMap<>();
                partials.put(CONTENT, resourceContent);

                Map<String, String> context = new HashMap<>(settings);
                context.put(ERROR_PARAMETER_NAME, request.getParameter(ERROR_PARAMETER_NAME));

                ThemeSettings themeSettings = configurationService.getThemeSettings();
                context.put(PRODUCT_COLOR, themeSettings.getHeaderAccentColor());
                context.put(INSTANCE_NAME, themeSettings.getHeaderName());

                MustacheRenderer mustacheRenderer = new MustacheRenderer(layout, partials, context);
                mustacheRenderer.render(response);
            } else if (request.getMethod().equals(POST)) {
                super.doFilter(request, response, chain);
            } else {
                throw new AuthenticationServiceException("Authentication method must be a POST or GET got: " + request.getMethod());
            }
        } else if (!request.getRequestURI().contains(BLACKLIST_RESOURCE)) {
            chain.doFilter(request, response);
        }
    }
}
