package com.xebialabs.xlrelease.api.internal;

import com.xebialabs.xlrelease.security.SessionService;
import com.xebialabs.xlrelease.spring.configuration.XlrProfiles;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.Context;
import jakarta.ws.rs.core.MediaType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Profile;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.oauth2.client.ClientAuthorizationException;
import org.springframework.security.oauth2.client.OAuth2AuthorizeRequest;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClient;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientManager;
import org.springframework.security.oauth2.core.OAuth2AccessToken;
import org.springframework.security.oauth2.core.OAuth2ErrorCodes;
import org.springframework.stereotype.Controller;

import java.io.IOException;

import static com.xebialabs.xlrelease.auth.oidc.config.OpenIdConnectConfig.OIDC_LOGIN_PATH_NAME;

@Path("/oidc")
@Profile(XlrProfiles.OIDC_AUTH)
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class OidcTokenResource {


    private final OAuth2AuthorizedClientManager authorizedClientManager;
    private final SessionService sessionService;

    @Autowired
    public OidcTokenResource(OAuth2AuthorizedClientManager authorizedClientManager, SessionService sessionService) {
        this.authorizedClientManager = authorizedClientManager;
        this.sessionService = sessionService;
    }

    @GET
    @Path("token")
    public OAuth2AccessToken token(@Context HttpServletRequest servletRequest, @Context HttpServletResponse servletResponse) throws IOException {
        Authentication principal = SecurityContextHolder.getContextHolderStrategy().getContext().getAuthentication();

        OAuth2AuthorizeRequest authorizeRequest = OAuth2AuthorizeRequest
                .withClientRegistrationId("xl-release")
                .principal(principal)
                .attribute(HttpServletRequest.class.getName(), servletRequest)
                .attribute(HttpServletResponse.class.getName(), servletResponse)
                .build();

        try {
            OAuth2AuthorizedClient authorizedClient = this.authorizedClientManager.authorize(authorizeRequest);
            return authorizedClient.getAccessToken();
        } catch (ClientAuthorizationException e) {
            String errorCode = e.getError().getErrorCode();
            if (OAuth2ErrorCodes.INVALID_GRANT.equalsIgnoreCase(errorCode) || OAuth2ErrorCodes.INVALID_TOKEN.equalsIgnoreCase(errorCode)) {
                this.sessionService.disconnect(principal.getName());
                servletResponse.sendRedirect(OIDC_LOGIN_PATH_NAME);
            }
            throw e;
        }

    }
}
