#
# Copyright 2017 XEBIALABS
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

import sys
import string
import com.xhaus.jyson.JysonCodec as json
from xlrelease.HttpRequest import HttpRequest
import urllib
from xlrelease.OAuthSupport import OAuthHttpClient
from azure.util import error

class AzureServer(object):
    """Exceptions are documented in the same way as classes.

    The __init__ method may be documented in either the class level
    docstring, or as a docstring on the __init__ method itself.

    Either form is acceptable, but the two should not be mixed. Choose one
    convention to document the __init__ method and be consistent with it.
    """

    __json_patch = 'application/json-patch+json'
    __json = 'application/json'

    def __init__(self, azure_server, project):
        if project is None:
            error('No project provided.')

        if azure_server is None:
            error('No server provided.')

        self.azure_url = "https://dev.azure.com/%s/%s/" % (azure_server['organization'], project)
        azure_server['url'] = self.azure_url

        if str(azure_server['authenticationMethod']) == 'OAuth2':
            access_token_url = "https://login.microsoftonline.com/%s/oauth2/v2.0/token" % azure_server['tenantId']
            azure_server['accessTokenUrl'] = access_token_url
            azure_server['password'] = AzureServer.get_access_token(azure_server)
            azure_server['authenticationMethod'] = 'Bearer'
        self.request = HttpRequest(azure_server)
         
    @staticmethod
    def get_access_token(httpConnection):
        body = dict(
            grant_type = "client_credentials",
            client_id = httpConnection['clientId'],
            client_secret = httpConnection['clientSecret'],
            scope = httpConnection['scope']
        )
        oauth_client = OAuthHttpClient(httpConnection)
        response = oauth_client.post(httpConnection['accessTokenUrl'], body=urllib.urlencode(body))
        response_body = json.loads(response.getResponse())
        if response.getStatus() == 200:
           return response_body['access_token']
        else:
           error("Failed to fetch OAuth access token", response)

    def createWorkItem(self, workItemType, workItemTitle, fields):
        if not workItemType:
            raise Exception("Work Item Type is a mandatory field.")

        if not workItemTitle:
            raise Exception("Work Item Title is a mandatory field.")

        items = [self._formatField("System.Title", workItemTitle)]

        for key, value in fields.iteritems():
            items.append(self._formatField(key, value))

        url = '_apis/wit/workitems/$%s?api-version=7.1-preview.3' % workItemType
        response = self.request.post(url, self._formatJsonContent(items), contentType=self.__json_patch)

        if response.status == 200:
            return json.loads(response.response)['id']
        else:
            error("Failed to create work item.", response)

    def getWorkItem(self, workItemId):
        if not workItemId:
            raise Exception("Work Item Id is a mandatory field.")

        response = self.request.get('_apis/wit/workitems/%s?api-version=7.1-preview.3' % workItemId)

        if not response.isSuccessful():
            error("Error fetching work item.", response)

        return json.loads(response.response)['fields']

    def getWorkItemState(self, workItemId):
        return self.getWorkItem(workItemId)["System.State"]

    def wiqlWorkItems(self, query):
        if not query:
            raise Exception("Query is a mandatory field.")

        content = '{ "query": %s }' %  json.dumps(query)

        response = self.request.post('_apis/wit/wiql?api-version=7.1-preview.2', content, contentType=self.__json)

        if response.status != 200:
            error("Failed to execute the query.", response)

        data = json.loads(response.response)

        ids = []
        for item in data['workItems']:
            ids.append(item['id'])

        return ids

    def queryWorkItems(self, queryName):
        if not queryName:
            raise Exception("Query Name is a mandatory field.")

        response = self.request.get('_apis/wit/queries/%s?api-version=7.1-preview.2&$expand=wiql' % queryName)

        if response.status != 200:
            error("Query with name '%s' not found." % queryName, response)

        wiql = json.loads(response.response)['wiql']
        return self.wiqlWorkItems(wiql)

    def getBuildDefinitionId(self, buildDefinitionName):
        if not buildDefinitionName:
            raise Exception("Build Definition Name is a mandatory field.")

        response = self.request.get('_apis/build/definitions?api-version=7.1-preview.7&name=%s' % buildDefinitionName)

        if response.status != 200:
            error("Error fetching build definition.", response)
        
        buildDefinition = json.loads(response.response)

        if buildDefinition["count"] == 0:
            error("Build definition with name %s was not found." % buildDefinitionName)

        return buildDefinition["value"][0]["id"]

    def queueBuild(self, buildDefinitionId, sourceBranch, buildProperties):
        if sourceBranch:
            sourceBranch = ', "sourceBranch": "'+ sourceBranch + '"'
        else:
            sourceBranch = ''
        if not buildProperties:
            params = ''
        else:
            mapSize = len(buildProperties)
            params = ', "parameters": "{'
            for buildProperty in buildProperties.keys():
                key = buildProperty
                value = buildProperties.get(buildProperty)
                params += "'" + key + "'" + ":"
                params += "'" + value + "'"
                mapSize -= 1
                if mapSize > 0:
                    params += ","
                else:
                    params += '}"'

        content = '{"definition": { "id": %s } %s %s }' % (buildDefinitionId, sourceBranch, params)
        response = self.request.post('_apis/build/builds?api-version=7.1-preview.7', content, contentType=self.__json)

        if response.status != 200:
            error("Failed to queue build.", response)

        return json.loads(response.response)["id"]

    def getBuild(self, buildId):
        response = self.request.get('_apis/build/builds/%s?api-version=7.1-preview.7' % buildId)

        if response.status != 200:
            error("Failed to fetch build status.", response)

        return json.loads(response.response)

    def addBuildPipelineTag(self, buildId, tag):
        if not buildId:
            raise Exception("Build Id is a mandatory field.")

        tag = json.dumps(tag).strip('"')
        response = self.request.put('_apis/build/builds/%s/tags/%s?api-version=7.1-preview.3' % (buildId, tag), "", contentType=self.__json)

        if response.status != 200:
            error("Failed to add build tag.", response)

    def updateWorkItemFields(self, workItemId, fields):
        if not workItemId:
            raise Exception("Work Item Id is a mandatory field.")

        items = []
        for key, value in fields.iteritems():
            items.append(self._formatField(key, value))

        response = self.request.patch('_apis/wit/workitems/%s?api-version=7.1-preview.3' % workItemId, self._formatJsonContent(items), contentType=self.__json_patch)
        
        if response.status != 200:
            error("Failed to update field(s).", response)

    def getChangesets(self):
        response = self.request.get('_apis/tfvc/changesets?api-version=7.1-preview.3')

        if response.status != 200:
            error("Failed to fetch changesets.", response)

        changeSets = json.loads(response.response)

        if changeSets["count"] > 0:
            return changeSets["value"]
    
    def getCommits(self, repositoryName, branchName):
        response = self.request.get('_apis/git/repositories/%s/commits?api-version=7.1-preview.1&searchCriteria.compareVersion.version=%s&searchCriteria.$top=2' % (repositoryName, branchName))

        if response.status != 200:
            error("Failed to fetch commits.", response)

        commits = json.loads(response.response)

        if commits["count"] > 0:
            return commits["value"]

    def _formatField(self, fieldName, fieldValue):
        if not fieldName or not fieldValue:
            raise Exception("Field name or value are not specified.")

        return'{"path": "/fields/%s", "value": %s, "op": "add"}' % (fieldName, json.dumps(fieldValue))

    def _formatJsonContent(self, items):
        if items:
            return ",".join(items).join(("[","]"))

    def getPiplineId(self, pipelineName):

        if not pipelineName:
            raise Exception("Pipeline Name is a mandatory field.")

        response = self.request.get(
            '_apis/build/definitions?api-version=7.1&name=%s' % (pipelineName))

        if response.status != 200:
            error("Error fetching pipeline.", response)

        pipeline = json.loads(response.response)

        if pipeline["count"] == 0:
            error("Pipeline with name %s was not found." % pipelineName)

        return pipeline["value"][0]["id"]

    def runPipeline(self, pipelineId, sourceBranch, templateParameters, variables):
        run_payload = {}
        if sourceBranch:
            run_payload["resources"] = {
                "repositories": {
                    "self": {
                        "refName": "{}".format(sourceBranch)
                    }
                }
            }

        if templateParameters:
            run_payload["templateParameters"] = templateParameters

        if variables:
            run_payload["variables"] = dict(
                (key, {"value": value}) for key, value in variables.items()
            )

        content = json.dumps(run_payload)
        response = self.request.post('_apis/pipelines/%s/runs?api-version=7.1' % (pipelineId),
                                     content, contentType=self.__json)
        if response.status != 200:
            error("Failed to run pipeline.", response)

        return json.loads(response.response)["id"]

    def getRunPipeline(self, pipelineId, pipelineRunId):
        response = self.request.get('_apis/build/builds/%s?api-version=7.1' % (pipelineRunId))
        if response.status != 200:
            error("Failed to fetch run pipeline status.", response)

        return json.loads(response.response)