/**
 * Copyright (c) 2023. All rights reserved.
 *
 * This software and all trademarks, trade names, and logos included herein are the property of Digital.ai Software, Inc. and its affiliates, subsidiaries, and licensors.
 */
package com.xebialabs.xlrelease.plugin.platform.connector

import com.xebialabs.deployit.booter.local.utils.Strings.{isNotBlank, isNotEmpty}
import com.xebialabs.xlrelease.domain.configuration.HttpConnection.AuthenticationMethod
import com.xebialabs.xlrelease.plugin.platform.ServerException
import com.xebialabs.xlrelease.plugin.platform.connector.HttpClientWrapper.{MULTIPART_BODY_FORM, RequestMediaType}
import grizzled.slf4j.Logging
import org.apache.hc.client5.http.auth.{AuthScope, UsernamePasswordCredentials}
import org.apache.hc.client5.http.config.ConnectionConfig
import org.apache.hc.client5.http.impl.auth.BasicCredentialsProvider
import org.apache.hc.client5.http.impl.classic.{CloseableHttpClient, HttpClientBuilder}
import org.apache.hc.client5.http.impl.io.BasicHttpClientConnectionManager
import org.apache.hc.core5.http.HttpHost
import org.springframework.core.io.ByteArrayResource
import org.springframework.http._
import org.springframework.http.client.{ClientHttpResponse, HttpComponentsClientHttpRequestFactory}
import org.springframework.util.LinkedMultiValueMap
import org.springframework.web.client.{ResponseErrorHandler, RestTemplate}
import org.springframework.web.util.UriComponentsBuilder

import java.net.URI
import java.util.concurrent.TimeUnit
import scala.jdk.CollectionConverters._
import scala.reflect.ClassTag
import scala.util.{Failure, Success, Try}


object HttpClientWrapper {

  def apply(configuration: ServerConfiguration): HttpClientWrapper = {
    new HttpClientWrapper(configuration.url, restTemplate(configuration)).withAuth(configuration)
  }

  private def restTemplate(serverConfiguration: ServerConfiguration): RestTemplate = {
    val httpClient = getDefaultClient(serverConfiguration)
    val httpComponentsClientHttpRequestFactory = new HttpComponentsClientHttpRequestFactory(httpClient)
    val restTemplate = new RestTemplate(httpComponentsClientHttpRequestFactory)
    restTemplate.setErrorHandler(LoggerResponseErrorHandler)
    restTemplate
  }

  def getDefaultClient(serverConfiguration: ServerConfiguration): CloseableHttpClient = {
    val httpClientBuilder = HttpClientBuilder.create()
    if (serverConfiguration.proxyConfiguration != null) {
      val proxyConfig = serverConfiguration.proxyConfiguration
      val proxyHost = proxyConfig.host
      val proxyPort = proxyConfig.port
      if (isNotBlank(proxyHost) && isNotBlank(proxyPort)) {
        val proxy = new HttpHost(proxyHost, proxyPort.toInt)
        httpClientBuilder.setProxy(proxy)
      }
      val proxyUsername = proxyConfig.username
      val proxyPassword = proxyConfig.password
      if (isNotBlank(proxyUsername) && isNotBlank(proxyPassword)) {
        val credentialProvider = new BasicCredentialsProvider()
        credentialProvider.setCredentials(
          new AuthScope(proxyHost, proxyPort.toInt),
          new UsernamePasswordCredentials(proxyUsername, proxyPassword.toCharArray)
        )
        httpClientBuilder.setDefaultCredentialsProvider(credentialProvider)
      }
    }

    val connectionTimeout = serverConfiguration.connectionConfiguration.timeout
    val cm = new BasicHttpClientConnectionManager()
    cm.setConnectionConfig(ConnectionConfig.custom()
      .setSocketTimeout(connectionTimeout, TimeUnit.SECONDS)
      .build())
    httpClientBuilder.setConnectionManager(cm)
    httpClientBuilder.disableCookieManagement()
    httpClientBuilder.build()
  }

  sealed trait RequestMediaType

  case class MULTIPART_BODY_FORM() extends RequestMediaType with Logging

  implicit class ResponseEntityExtension[T](response: ResponseEntity[T]) extends Logging {
    def tried: Try[T] = {
      if (response.isSuccessful) {
        Success(response.getBody)
      } else {
        Failure {
          response.getStatusCode match {
            case code@HttpStatus.UNAUTHORIZED => ServerException("Invalid credentials", statusCode = code.value())
            case code@HttpStatus.NOT_FOUND => ServerException("resource not found", statusCode = code.value())
            case _ => ServerException("Unable to connect to the Delivery Insights Service", statusCode = response.getStatusCodeValue)
          }
        }
      }
    }

    def checkResponse(errMsg: String): Try[ResponseEntity[T]] = {
      if (response.isSuccessful) {
        Success(response)
      } else {
        Failure(ServerException(s"$errMsg ${response.toString}"))
      }
    }

    def isSuccessful: Boolean = response.getStatusCode.is2xxSuccessful()
  }

}

class HttpClientRequest(restTemplate: RestTemplate, url: URI, httpMethod: HttpMethod, entity: HttpEntity[_]) {
  def doRequest[T: ClassTag](): Try[ResponseEntity[T]] = Try {
    restTemplate.exchange(url, httpMethod, entity, implicitly[ClassTag[T]].runtimeClass).asInstanceOf[ResponseEntity[T]]
  }
}


class HttpClientWrapper(restApiUrl: String, val restTemplate: RestTemplate) {
  private val formData = new LinkedMultiValueMap[String, AnyRef]
  private val uriComponentsBuilder = UriComponentsBuilder.fromHttpUrl(restApiUrl)
  private val headers = new HttpHeaders()
  private var mediaType: RequestMediaType = _
  private var uriEncoded: Boolean = false

  def setType(mediaType: RequestMediaType): HttpClientWrapper = {
    this.mediaType = mediaType
    this
  }

  def setUriEncoded(uriEncoded: Boolean): HttpClientWrapper = {
    this.uriEncoded = uriEncoded
    this
  }

  def addFormData(name: String, value: String): HttpClientWrapper = {
    formData.add(name, value)
    this
  }

  def addFile(path: String, fileName: String, content: () => Array[Byte]): HttpClientWrapper = {
    val resource = new ByteArrayResource(content()) {
      override def getFilename: String = fileName
    }
    formData.add(path, resource)
    this
  }

  def addQueryParam(name: String, value: String): HttpClientWrapper = {
    uriComponentsBuilder.queryParam(name, value)
    this
  }

  def accept(contentType: MediaType): HttpClientWrapper = {
    headers.setAccept(List(contentType).asJava)
    this
  }

  def withAuth(serverConfiguration: ServerConfiguration): HttpClientWrapper = {
    val token = serverConfiguration.apiToken
    if (isNotEmpty(token)) {
      headers.set(HttpHeaders.AUTHORIZATION, getBasicAuthorization(token))
    }
    this
  }

  private def getBasicAuthorization(token: String): String = {
    s"Token $token"
  }

  def request[T](method: String, path: String, query: String, body: T = null): HttpClientRequest = {
    uriComponentsBuilder.path(path)
    uriComponentsBuilder.query(query)
    new HttpClientRequest(restTemplate, uriComponentsBuilder.build(uriEncoded).toUri, org.springframework.http.HttpMethod.valueOf(method), createHttpEntity(body))
  }

  def post[T](path: String, body: T = null): HttpClientRequest = {
    uriComponentsBuilder.path(path)
    new HttpClientRequest(restTemplate, uriComponentsBuilder.build(uriEncoded).toUri, HttpMethod.POST, createHttpEntity(body))
  }

  def get(path: String): HttpClientRequest = {
    uriComponentsBuilder.path(path)
    new HttpClientRequest(restTemplate, uriComponentsBuilder.build(uriEncoded).toUri, HttpMethod.GET, createHttpEntity())
  }

  def head(path: String): HttpClientRequest = {
    uriComponentsBuilder.path(path)
    new HttpClientRequest(restTemplate, uriComponentsBuilder.build(uriEncoded).toUri, HttpMethod.HEAD, createHttpEntity())
  }

  private def createHttpEntity[B](body: B = null): HttpEntity[_ >: LinkedMultiValueMap[String, AnyRef] with B] = {

    val entity = mediaType match {
      case MULTIPART_BODY_FORM() => headers.setContentType(MediaType.MULTIPART_FORM_DATA)
        new HttpEntity(formData, headers)
      case _ => headers.setContentType(MediaType.APPLICATION_JSON)
        new HttpEntity(body, headers)
    }
    entity

  }

}

object LoggerResponseErrorHandler extends ResponseErrorHandler with Logging {
  override def hasError(response: ClientHttpResponse): Boolean = false

  override def handleError(response: ClientHttpResponse): Unit = {
    logger.error("Error response from Server: ")
    logger.error(response.getBody)
  }
}

case class ProxyConfiguration(host: String,
                              port: String,
                              username: String,
                              password: String,
                              domain: String
                             ) {
  def withPassword(password: String): ProxyConfiguration = {
    copy(password = password)
  }
}

case class ConnectionConfiguration(authenticationMethod: AuthenticationMethod = AuthenticationMethod.Basic,
                                   timeout: Int = 60
                                  )

case class ServerConfiguration(url: String,
                               username: String,
                               password: String = null,
                               apiToken: String,
                               proxyConfiguration: ProxyConfiguration = null,
                               connectionConfiguration: ConnectionConfiguration = ConnectionConfiguration()
                              )
