/**
 * Copyright (c) 2023. All rights reserved.
 *
 * This software and all trademarks, trade names, and logos included herein are the property of Digital.ai Software, Inc. and its affiliates, subsidiaries, and licensors.
 */
package com.xebialabs.xlrelease.plugin.platform.converter

import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.plugin.platform.utils.StringUtils._
import com.xebialabs.xlrelease.plugin.platform.{AgilityPluginConfigurationRequest, BitbucketCloudPluginConfigurationRequest, BitbucketServerPluginConfigurationRequest, GithubPluginConfigurationRequest, GitlabPluginConfigurationRequest, JenkinsPluginConfigurationRequest, JiraPluginConfigurationRequest, PluginConfigurationRequest}
import grizzled.slf4j.Logging

object BaseConfigurationConverter extends Logging {

  def toPluginConfigurationRequest(conf: BaseConfiguration, overwrite: Boolean): Option[PluginConfigurationRequest] = {
      conf.getType.toString match {
        case "jira.Server" =>
          val apiToken = conf.getProperty[String]("apiToken")
          Some(JiraPluginConfigurationRequest(
            name = conf.getId,
            description = conf.getTitle,
            user = conf.getProperty[String]("username"),
            password = apiToken.decrypt,
            url = conf.getProperty[String]("url"),
            improve_types = "", // TODO
            maintain_types = "",// TODO
            team_id = null,
            overwrite = overwrite,
          ))
        case "jenkins.Server" =>
          Some(JenkinsPluginConfigurationRequest(
            name = conf.getId,
            description = conf.getTitle,
            user = conf.getProperty[String]("username"),
            password = conf.getProperty[String]("password").decrypt,
            api_token = conf.getProperty[String]("apiToken").decrypt,
            url = conf.getProperty[String]("url"),
            team_id = null,
            overwrite = overwrite,
          ))
        case "github.Server" =>
          Some(GithubPluginConfigurationRequest(
            name = conf.getId,
            description = conf.getTitle,
            owner = "",
            url = conf.getProperty[String]("baseUrl"),
            api_token = conf.getProperty[String]("loginOrToken").decrypt,
            overwrite = overwrite,
          ))
        case "gitlab.Server" =>
          Some(GitlabPluginConfigurationRequest(
            name = conf.getId,
            description = conf.getTitle,
            url = conf.getProperty[String]("url"),
            api_token = conf.getProperty[String]("api_key").decrypt,
            overwrite = overwrite,
          ))
        case "versionone.Server" =>
          Some(AgilityPluginConfigurationRequest(
            name = conf.getId,
            description = conf.getTitle,
            url = conf.getProperty[String]("url"),
            api_token = conf.getProperty[String]("accessToken").decrypt,
            overwrite = overwrite,
          ))
        case "bitbucket.Cloud" =>
          Some(BitbucketCloudPluginConfigurationRequest(
            name = conf.getId,
            description = conf.getTitle,
            url = conf.getProperty[String]("url"),
            user = conf.getProperty[String]("username"),
            password = conf.getProperty[String]("password").decrypt,
            overwrite = overwrite,
          ))
        case "bitbucket.Server" =>
          Some(BitbucketServerPluginConfigurationRequest(
            name = conf.getId,
            description = conf.getTitle,
            url = conf.getProperty[String]("url"),
            user = conf.getProperty[String]("username"),
            password = conf.getProperty[String]("password").decrypt,
            overwrite = overwrite,
          ))
        case _ =>
          logger.warn(s"Configuration type [${conf.getType}] doesn't have mapping for request")
          None

      }

  }

}
