/**
 * Copyright (c) 2023. All rights reserved.
 *
 * This software and all trademarks, trade names, and logos included herein are the property of Digital.ai Software, Inc. and its affiliates, subsidiaries, and licensors.
 */
package com.xebialabs.xlrelease.plugin.platform.listener

import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{AsyncSubscribe, XLReleaseEventListener}
import com.xebialabs.xlrelease.plugin.platform.service.ReleaseEventService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired

class ReleaseEventsListener extends XLReleaseEventListener with Logging {

  @Autowired
  var releaseEventsService: ReleaseEventService = _

  @AsyncSubscribe
  def onTaskStartedEvent(event: TaskStartedEvent): Unit = {
    val release = event.task.getPhase.getRelease
    val info = s"Task started event request for [release = ${release.getId} and task status =  ${event.task.getStatus}]"
    releaseEventsService.processRelease(info, release)
  }

  @AsyncSubscribe
  def onTaskCompletedEvent(event: TaskCompletedEvent): Unit = {
    val release = event.task.getPhase.getRelease
    val info = s"Task completed event request for [release = ${release.getId} and task status =  ${event.task.getStatus}]"
    releaseEventsService.processRelease(info, release)
  }

  @AsyncSubscribe
  def onTaskFailedEvent(event: TaskFailedEvent): Unit = {
    val release = event.task.getPhase.getRelease
    val info = s"Task failed event request for [release = ${release.getId} and task status =  ${event.task.getStatus}]"
    releaseEventsService.processRelease(info, release)
  }

  @AsyncSubscribe
  def onTaskRetriedEvent(event: TaskRetriedEvent): Unit = {
    val release = event.task.getPhase.getRelease
    val info = s"Task retried event request for [release = ${release.getId} and task status =  ${event.task.getStatus}]"
    releaseEventsService.processRelease(info, release)
  }

  @AsyncSubscribe
  def onTaskSkippedEvent(event: TaskSkippedEvent): Unit = {
    val release = event.task.getPhase.getRelease
    val info = s"Task skipped event request for [release = ${release.getId} and task status =  ${event.task.getStatus}]"
    releaseEventsService.processRelease(info, release)
  }

  @AsyncSubscribe
  def onTaskAbortedEvent(event: TaskAbortedEvent): Unit = {
    val release = event.task.getPhase.getRelease
    val info = s"Task aborted event request for [release = ${release.getId} and task status =  ${event.task.getStatus}]"
    releaseEventsService.processRelease(info, release)
  }

  @AsyncSubscribe
  def onReleaseStartedEvent(event: ReleaseStartedEvent): Unit = {
    val release = event.release
    val info = s"Release started event request for [release = ${release.getId} and release status =  ${release.getStatus}]"
    releaseEventsService.processRelease(info, release)
  }

  @AsyncSubscribe
  def onReleaseFailedEvent(event: ReleaseFailedEvent): Unit = {
    val release = event.release
    val info = s"Release failed event request for [release = ${release.getId} and release status =  ${release.getStatus}]"
    releaseEventsService.processRelease(info, release)
  }

  @AsyncSubscribe
  def onReleaseRestartedEvent(event: ReleaseRestoredEvent): Unit = {
    val info = s"Release restarted event request for [${event.updated.getId} and release status =  ${event.updated.getStatus}]"
    releaseEventsService.processRelease(info, event.updated)
  }

  @AsyncSubscribe
  def onReleaseAbortedEvent(event: ReleaseAbortedEvent): Unit = {
    val release = event.release
    val info = s"Release aborted event request for [release = ${release.getId} and release status =  ${release.getStatus}] "
    releaseEventsService.processRelease(info, release)
  }

  @AsyncSubscribe
  def onReleaseUpdatedEvent(event: ReleaseUpdatedEvent): Unit = {
    val release = event.updated
    val info = s"Release updated event request for [${release.getId} and release status =  ${release.getStatus}]"
    releaseEventsService.processRelease(info, release)
  }

  @AsyncSubscribe
  def onSharedConfigurationCreated(event: ConfigurationCreatedEvent): Unit = {
    releaseEventsService.processSharedConfiguration(event.conf)
  }

  @AsyncSubscribe
  def onSharedConfigurationUpdated(event: ConfigurationUpdatedEvent): Unit = {
    releaseEventsService.processSharedConfiguration(event.updated, overwrite = true)
  }

  @AsyncSubscribe
  def onReleaseCompletedEvent(event: ReleaseCompletedEvent): Unit = {
    val release = event.release
    val info = s"Release completed event request for [release = ${release.getId} and release status =  ${release.getStatus}]"
    releaseEventsService.processRelease(info, release)
  }

}
