/**
 * Copyright (c) 2023. All rights reserved.
 *
 * This software and all trademarks, trade names, and logos included herein are the property of Digital.ai Software, Inc. and its affiliates, subsidiaries, and licensors.
 */
package com.xebialabs.xlrelease.plugin.platform

import com.fasterxml.jackson.annotation.{JsonCreator, JsonProperty}
import com.fasterxml.jackson.databind.annotation.JsonDeserialize
import com.xebialabs.xlrelease.plugin.platform.domain.PlatformServer
import com.xebialabs.xlrelease.plugin.platform.utils.StringUtils.UtilsString
import com.xebialabs.xlrelease.plugins.dashboard.service.TileService
import com.xebialabs.xlrelease.serialization.json.jackson.ConfigurationIdDeserializer
import com.xebialabs.xlrelease.service.{ConfigurationVariableService, ReleaseService, SharedConfigurationService}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller
import org.springframework.web.client.HttpClientErrorException
import org.springframework.web.util.UriUtils
import java.util.stream.Collectors
import jakarta.servlet.http.HttpServletRequest
import jakarta.ws.rs._
import jakarta.ws.rs.core.{Context, HttpHeaders, MediaType, Response, UriInfo}
import scala.beans.BeanProperty


@Path("/platform")
@Produces(Array(MediaType.APPLICATION_JSON))
@Consumes(Array(MediaType.APPLICATION_JSON))
@Controller
class PlatformResource @Autowired()(
                                     val releaseService: ReleaseService,
                                     val tilesService: TileService,
                                     val sharedConfigurationService: SharedConfigurationService,
                                     val configurationVariableService: ConfigurationVariableService) extends Logging {


  @GET
  @Path("/proxy/{platformId:.*Configuration[^/-]*}/{url:.*}")
  def proxyRequestGET(@PathParam("platformId") platformId: String,
                      @PathParam("url") url: String,
                      @Context uriInfo: UriInfo,
                      @Context request: HttpServletRequest,
                      @Context httpheaders: HttpHeaders): Response = {
    doProxy(platformId, url, uriInfo, request, httpheaders, null)
  }

  @POST
  @Path("/proxy/{platformId:.*Configuration[^/-]*}/{url:.*}")
  def proxyRequestPOST(@PathParam("platformId") platformId: String,
                       @PathParam("url") url: String,
                       @Context uriInfo: UriInfo,
                       @Context request: HttpServletRequest,
                       @Context httpheaders: HttpHeaders,
                       body: String): Response = {
    doProxy(platformId, url, uriInfo, request, httpheaders, body)
  }

  @PUT
  @Path("/proxy/{platformId:.*Configuration[^/-]*}/{url:.*}")
  def proxyRequestPUT(@PathParam("platformId") platformId: String,
                      @PathParam("url") url: String,
                      @Context uriInfo: UriInfo,
                      @Context request: HttpServletRequest,
                      @Context httpheaders: HttpHeaders,
                      body: String): Response = {
    doProxy(platformId, url, uriInfo, request, httpheaders, body)
  }

  @PATCH
  @Path("/proxy/{platformId:.*Configuration[^/-]*}/{url:.*}")
  def proxyRequestPATCH(@PathParam("platformId") platformId: String,
                        @PathParam("url") url: String,
                        @Context uriInfo: UriInfo,
                        @Context request: HttpServletRequest,
                        @Context httpheaders: HttpHeaders,
                        body: String): Response = {
    doProxy(platformId, url, uriInfo, request, httpheaders, body)
  }
  @DELETE
  @Path("/proxy/{platformId:.*Configuration[^/-]*}/{url:.*}")
  def proxyRequestDELETE(@PathParam("platformId") platformId: String,
                         @PathParam("url") url: String,
                         @Context uriInfo: UriInfo,
                         @Context request: HttpServletRequest,
                         @Context httpheaders: HttpHeaders): Response = {
    doProxy(platformId, url, uriInfo, request, httpheaders, body=null)
  }

  private def doProxy(platformId: String,
                      url: String,
                      uriInfo: UriInfo,
                      request: HttpServletRequest,
                      httpheaders: HttpHeaders,
                      body: String): Response = {
    val platformServer = findCtmServer(platformId)
    // @Path annotation will always remove the trailing slash so we have to check if it was originally there and add it manually to the url
    val path = uriInfo.getRequestUri.getPath
    val urlEnding = if (path.charAt(path.length - 1) == '/') '/' else ""
    // get the original query parameters from the url and encode them
    val platformQuery = UriUtils.encodeQuery(uriInfo.getRequestUri.getQuery, "UTF-8")
    val platformPath = s"/$url$urlEnding"

    try {
      val resp = platformServer.getClient.getWrapper
        .request(request.getMethod.toUpperCase, platformPath, platformQuery,body)
        .doRequest[String]().get
      createResponse(resp.getStatusCode.value, resp.getBody, resp.getHeaders)
    } catch {
      case e: HttpClientErrorException =>
        logger.error("Error while connecting to platform", e)
        createResponse(e.getStatusCode.value, e.getResponseBodyAsString, e.getResponseHeaders)
    }
  }

  private def createResponse(statusCode: Int, body: String, headers: org.springframework.http.HttpHeaders): Response = {
    val responseBuilder = Response.status(statusCode).entity(body)
    headers.forEach((h, v) => {
      responseBuilder.header(h, v.stream().map(_.toString).collect(Collectors.joining(",")))
    })
    responseBuilder.build()
  }

  @GET
  @Path("/server/{tileId:.*Tile[^/-]*}")
  def getServer(@PathParam("tileId") tileId: String): ServerResponse = {
    val tile = tilesService.findTileById(tileId)
    val server = tile.getProperty[PlatformServer]("server")
    ServerResponse(
      serverUrl = server.getUrl,
      apiToken = server.getApiToken.decrypt
    )
  }

  @POST
  @Path("/progression")
  def addProgression(createProgression: CreateProgression): Unit = {
    val platformServer = findCtmServer(createProgression.server)
    val client = Client.apply(platformServer)
    client.createProgression(CreateProgressionRequest(createProgression.progressionName))
  }

  private def findCtmServer(id: String): PlatformServer = {
    val platformServer = sharedConfigurationService.findById(id).asInstanceOf[PlatformServer]
    configurationVariableService.resolve(platformServer)
    platformServer
  }

}

case class ServerResponse(@BeanProperty serverUrl: String,
                          @BeanProperty apiToken: String)

case class CreateProgression(@BeanProperty
                             @JsonDeserialize(using = classOf[ConfigurationIdDeserializer]) @JsonProperty("server") server: String,
                             @BeanProperty @JsonProperty("progressionName") progressionName: String) {
  @JsonCreator() def this() = this("", "")
}
